/*FOCUS-2.0 is a sampling and statistical data analysis program
for spatial data sets. Copyright (C) 2003  Lutz Tischendorf

This program is free software; you can redistribute it
and/or modify it under the terms of the GNU General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public
License along with this program; if not, write to the Free
Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.*/

#ifndef __OBSERVER_CPP
#define __OBSERVER_CPP

#include "observer.h"
#include "focusregistry.h"
#include "controller.h"
#include "logging.h"
#include "focusutil.h"
#include "statistics.h"
#include "plot.h"
#include <math.h>

Observer::Observer(FocusRegistry* _registry, const char* _name, unsigned short _runs)
{
	registry = _registry;	
	name = _name;
	stats = false;
	runs = _runs;         		
}

Observer::~Observer()
{
}

bool Observer::operator == (const Observer& _observer) const
{
	return 	name == _observer.name &&
			runs == _observer.runs;
}

bool Observer::operator!=(const Observer& _observer) const
{
	return !((*this)==_observer);
}

//***************************************************************************************************************

AnovaObserver::AnovaObserver(FocusRegistry* _registry, unsigned short _runs)
				   : Observer(_registry, "AnovaObserver", _runs)
{
	sst_observations = NULL;
	ssr_observations = NULL;
	sse_observations = NULL;
	mse_observations = NULL;
	f_observations = NULL;
	rr_observations = NULL;
	r_observations = NULL;
	plot_observations = NULL;
	initial_runs = _runs;

	init();	
}

void AnovaObserver::init()
{
	runs = initial_runs;

	if(sst_observations)
		delete sst_observations;
	sst_observations = new floatlist(runs);
	if(ssr_observations)
		delete ssr_observations;
	ssr_observations = new floatlist(runs);
	if(sse_observations)
		delete sse_observations;
	sse_observations = new floatlist(runs);
	if(mse_observations)
		delete mse_observations;
	mse_observations = new floatlist(runs);
	if(f_observations)
		delete f_observations;
	f_observations = new floatlist(runs);
	if(rr_observations)
		delete rr_observations;
	rr_observations = new floatlist(runs);
	if(r_observations)
		delete r_observations;
	r_observations = new floatlist(runs);
	if(plot_observations)
		delete plot_observations;
	plot_observations = new floatlist(runs);
}

AnovaObserver::~AnovaObserver()
{
	Observer::~Observer();
	delete sst_observations;
	delete ssr_observations;
	delete sse_observations;
	delete mse_observations;
	delete f_observations;
	delete rr_observations;
	delete r_observations;
	delete plot_observations;
}

void AnovaObserver::reset()
{
	stats = false;
}

void AnovaObserver::observe(int _run)
{
	AnovaData* anov_dat = registry->getController()->getAnovaData();
	if(anov_dat && _run<=runs && !stats)
	{
   		(*sst_observations)[_run]=anov_dat->get_sst();
		(*ssr_observations)[_run]=anov_dat->get_ssr();
		(*sse_observations)[_run]=anov_dat->get_sse();
		(*mse_observations)[_run]=anov_dat->get_mse();
		(*f_observations)[_run]=anov_dat->get_f();
		(*rr_observations)[_run]=anov_dat->get_rr();
		(*r_observations)[_run]=anov_dat->get_r();
		(*plot_observations)[_run]=anov_dat->get_n();
	}
	else
		Logging::log_debug("AnovaObserver::observe(", Utility::int_to_char(_run), ") no Anova Data found, or run out of range!",NULL);
}

void AnovaObserver::calc_stats()
{
	if(!stats)
	{
		floatlist* sst_stats = new floatlist(STATSVAR);
		floatlist* ssr_stats = new floatlist(STATSVAR);
		floatlist* sse_stats = new floatlist(STATSVAR);
		floatlist* mse_stats = new floatlist(STATSVAR);
		floatlist* f_stats = new floatlist(STATSVAR);
		floatlist* rr_stats = new floatlist(STATSVAR);
		floatlist* r_stats = new floatlist(STATSVAR); 
		floatlist* plot_stats = new floatlist(STATSVAR);

		(*sst_stats)[MEAN] = (*sst_observations).mean();
		(*sst_stats)[MIN] = (*sst_observations).min();
		(*sst_stats)[MAX] = (*sst_observations).max();
		(*sst_stats)[VAR] = (*sst_observations).varianz();
		(*sst_stats)[CONFID] = (*sst_observations).confidence_interval();
		(*sst_stats)[STDERR] = (*sst_observations).stderro();

		(*ssr_stats)[MEAN] = (*ssr_observations).mean();
		(*ssr_stats)[MIN] = (*ssr_observations).min();
		(*ssr_stats)[MAX] = (*ssr_observations).max();
		(*ssr_stats)[VAR] = (*ssr_observations).varianz();
		(*ssr_stats)[CONFID] = (*ssr_observations).confidence_interval();
		(*ssr_stats)[STDERR] = (*ssr_observations).stderro();

		(*sse_stats)[MEAN] = (*sse_observations).mean();
		(*sse_stats)[MIN] = (*sse_observations).min();
		(*sse_stats)[MAX] = (*sse_observations).max();
		(*sse_stats)[VAR] = (*sse_observations).varianz();
		(*sse_stats)[CONFID] = (*sse_observations).confidence_interval();
		(*sse_stats)[STDERR] = (*sse_observations).stderro();

		(*mse_stats)[MEAN] = (*mse_observations).mean();
		(*mse_stats)[MIN] = (*mse_observations).min();
		(*mse_stats)[MAX] = (*mse_observations).max();
		(*mse_stats)[VAR] = (*mse_observations).varianz();
		(*mse_stats)[CONFID] = (*mse_observations).confidence_interval();
		(*mse_stats)[STDERR] = (*mse_observations).stderro();

		(*f_stats)[MEAN] = (*f_observations).mean();
		(*f_stats)[MIN] = (*f_observations).min();
		(*f_stats)[MAX] = (*f_observations).max();
		(*f_stats)[VAR] = (*f_observations).varianz();
		(*f_stats)[CONFID] = (*f_observations).confidence_interval();
		(*f_stats)[STDERR] = (*f_observations).stderro();

		(*rr_stats)[MEAN] = (*rr_observations).mean();
		(*rr_stats)[MIN] = (*rr_observations).min();
		(*rr_stats)[MAX] = (*rr_observations).max();
		(*rr_stats)[VAR] = (*rr_observations).varianz();
		(*rr_stats)[CONFID] = (*rr_observations).confidence_interval();
		(*rr_stats)[STDERR] = (*rr_observations).stderro();

		(*r_stats)[MEAN] = (*r_observations).mean();
		(*r_stats)[MIN] = (*r_observations).min();
		(*r_stats)[MAX] = (*r_observations).max();
		(*r_stats)[VAR] = (*r_observations).varianz();
		(*r_stats)[CONFID] = (*r_observations).confidence_interval();
		(*r_stats)[STDERR] = (*r_observations).stderro();

		(*plot_stats)[MEAN] = (*plot_observations).mean();
		(*plot_stats)[MIN] = (*plot_observations).min();
		(*plot_stats)[MAX] = (*plot_observations).max();
		(*plot_stats)[VAR] = (*plot_observations).varianz();
		(*plot_stats)[CONFID] = (*plot_observations).confidence_interval();
		(*plot_stats)[STDERR] = (*plot_observations).stderro();

		//update actual runs, which might be reduced by zero regressions
		runs = (*plot_observations).entries();

		delete sst_observations;
		sst_observations = sst_stats;

		delete ssr_observations;
		ssr_observations = ssr_stats;

		delete sse_observations;
		sse_observations = sse_stats;

		delete mse_observations;
		mse_observations = mse_stats;

		delete f_observations;
		f_observations = f_stats;

		delete rr_observations;
		rr_observations = rr_stats;

		delete r_observations;
		r_observations = r_stats;

		delete plot_observations;
		plot_observations = plot_stats;

		stats = true;
	}
	else
		Logging::log_debug("AnovaObserver::calc_stats() statistics were already calculated!",NULL);	
}

void AnovaObserver::report_variable_names(ostream& strm) const
{
	strm<<"RUNS,";
	strm<<"PLT-MEAN,PLT-MIN,PLT-MAX,PLT-VAR,PLT-KON,PLT-SDE,";
	strm<<"SST-MEAN,SST-MIN,SST-MAX,SST-VAR,SST-KON,SST-SDE,";
	strm<<"SSR-MEAN,SSR-MIN,SSR-MAX,SSR-VAR,SSR-KON,SSR-SDE,";
	strm<<"SSE-MEAN,SSE-MIN,SSE-MAX,SSE-VAR,SSE-KON,SSE-SDE,";
	strm<<"MSE-MEAN,MSE-MIN,MSE-MAX,MSE-VAR,MSE-KON,MSE-SDE,";
	strm<<"F-MEAN,F-MIN,F-MAX,F-VAR,F-KON,F-SDE,";
	strm<<"RR-MEAN,RR-MIN,RR-MAX,RR-VAR,RR-KON,RR-SDE,";
	strm<<"R-MEAN,R-MIN,R-MAX,R-VAR,R-KON,R-SDE,";
}

void AnovaObserver::report(ostream& strm) const
{
	strm<<runs<<",";
	plot_observations->report(strm);
	sst_observations->report(strm);
	ssr_observations->report(strm);
	sse_observations->report(strm);
	mse_observations->report(strm);
	f_observations->report(strm);
	rr_observations->report(strm);
	r_observations->report(strm);
}

//***************************************************************************************************************

FunctionalStabilityObserver::FunctionalStabilityObserver(FocusRegistry* _registry, unsigned short _max_order, unsigned short _percentage)
							 : Observer(_registry, "FunctionalStabilityObserver", 0)
{
	max_order = _max_order;
	percentage = _percentage;
	order_observations = NULL;
	observations = 0;
	init();
}

FunctionalStabilityObserver::~FunctionalStabilityObserver()
{
	Observer::~Observer();
	delete order_observations;
}

void FunctionalStabilityObserver::reset()
{
	observations = 0;;
}

void FunctionalStabilityObserver::init()
{
	if(order_observations)
		delete order_observations;
	order_observations = new intlist(max_order);
}

void FunctionalStabilityObserver::observe(int _run)
{
	unsigned short order = registry->getController()->getMaxOrder();

	if(order == NULL_SHORT)
		return;

	if(order > 0 && order <= max_order)
	{
		(*order_observations)[order-1]++;
		observations++;
	}
	else
		Logging::log_debug("FunctionalStabilityObserver::observe(", Utility::int_to_char(_run), ") order out of range!",NULL);
}

void FunctionalStabilityObserver::report_variable_names(ostream& strm) const
{
	strm<<"FS_PERC,FS_OBS,";
	for(int i=1; i<=max_order; i++)
		strm<<"ORDER_"<<i<<",";
}

void FunctionalStabilityObserver::report(ostream& strm) const
{
	strm<<percentage<<",";
	strm<<observations<<",";
	order_observations->report(strm);
}

//***************************************************************************************************************

PlotObserver::PlotObserver(FocusRegistry* _registry, unsigned short _max_plots, unsigned short _runs, unsigned short _constraints)
							 : Observer(_registry, "PlotObserver", _runs)
{
	max_plots = _max_plots;
	number_of_constraints = _constraints;
	constraint_values = new floatlist(_constraints);
	resp_plot_id_observations = new dintcube(max_plots, runs, number_of_constraints);
	pred_plot_id_observations = new dintcube(max_plots, runs, number_of_constraints);
	plot_pred_observations = new dfloatcube(max_plots, runs, number_of_constraints);
	plot_resp_observations = new dfloatcube(max_plots, runs, number_of_constraints);
	plot_set_type = NULL;
}

PlotObserver::~PlotObserver()
{
	Observer::~Observer();
	delete resp_plot_id_observations;
	delete pred_plot_id_observations;
	delete plot_pred_observations;
	delete plot_resp_observations;
	delete constraint_values;
}

void PlotObserver::observe(int _run)
{
	if(!plot_set_type)
	{
		plotset_types ps_type = registry->getController()->getPlotSetType();
		plot_set_type = (ps_type == PLOT_BUFFER ? PS_PLOT_BUFFER : PS_PLOT_PLOT_PREDICTOR);
	}
	if(!strcmp(plot_set_type, PS_PLOT_BUFFER))
	{
		PlotList* plots = registry->getPlotSetList()->getPlotSetForReferenceName(PS_PLOT_BUFFER)->getPlotList();

		if(!plots || _run > runs)
		{
			Logging::log_debug("PlotObserver::observe(", Utility::int_to_char(_run), ") plotlist undefined!",NULL);
			return;
		}

		double constraint_value = ((DistanceConstraint*)registry->getController()->getConstraint())->getDistance();
		unsigned short constraint_index = registry->getController()->getConstraintIndex();
		unsigned short predictor_index = registry->getController()->getPredictorIndex();
		
		if(constraint_index <= number_of_constraints)
			(*constraint_values)[constraint_index-1] = constraint_value;
		else
			Logging::log_debug("PlotObserver::observe() constraint_index: ", Utility::int_to_char(constraint_index)," >  MaxConstraints!",NULL);

		int index = 0;
		PlotList::iterator plot_iterator = plots->begin();
		while(plot_iterator != plots->end())
		{
			Plot* tmpPlot = *plot_iterator++;
			if(tmpPlot->isSelected())
			{	
				if(index < max_plots)
				{
					(*pred_plot_id_observations)[constraint_index-1][_run][index] = atoi(tmpPlot->getId());
					(*plot_pred_observations)[constraint_index-1][_run][index] = tmpPlot->getPredictorVariableAtIndex(predictor_index).getValue();
					(*plot_resp_observations)[constraint_index-1][_run][index] = tmpPlot->getResponseVariableAtIndex(0).getValue();
					index++;
				}
				else
					Logging::log_debug("PlotObserver::observe() index: ", Utility::int_to_char(index)," >  MaxPlots!",NULL);
			}
		}
	}
	else
	{
		PlotList* pred_plots = registry->getPlotSetList()->getPlotSetForReferenceName(PS_PLOT_PLOT_PREDICTOR)->getPlotList();
		PlotList* resp_plots = registry->getPlotSetList()->getPlotSetForReferenceName(PS_PLOT_PLOT_RESPONSE)->getPlotList();

		if(!pred_plots || !resp_plots || _run > runs)
		{
			Logging::log_debug("PlotObserver::observe(", Utility::int_to_char(_run), ") plotlist undefined!",NULL);
			return;
		}

		double constraint_value = ((DistanceConstraint*)registry->getController()->getConstraint())->getDistance();
		unsigned short constraint_index = registry->getController()->getConstraintIndex();
		int current_plots = registry->getController()->getNumberOfCurrentSelectedPlots();
		
		if(constraint_index <= number_of_constraints)
			(*constraint_values)[constraint_index-1] = constraint_value;
		else
			Logging::log_debug("PlotObserver::observe() constraint_index: ", Utility::int_to_char(constraint_index)," >  MaxConstraints!",NULL);

		int index = 0;
		PlotList::iterator predplot_iterator = pred_plots->begin();
		while(predplot_iterator != pred_plots->end())
		{
			Plot* tmpPlot = *predplot_iterator++;
			if(tmpPlot->isSelected())
			{	
				if(index < max_plots && index < current_plots)
				{
					(*pred_plot_id_observations)[constraint_index-1][_run][index] = atoi(tmpPlot->getId());
					(*plot_pred_observations)[constraint_index-1][_run][index] = tmpPlot->getPredictorVariableAtIndex(0).getValue();
					index++;
				}
				else
					Logging::log_debug("PlotObserver::observe() index: ", Utility::int_to_char(index)," >  MaxPlots!",NULL);
			}
		}

		index = 0;
		PlotList::iterator respplot_iterator = resp_plots->begin();
		while(respplot_iterator != resp_plots->end())
		{
			Plot* tmpPlot = *respplot_iterator++;
			if(tmpPlot->isSelected())
			{	
				if(index < max_plots && index < current_plots)
				{
					(*resp_plot_id_observations)[constraint_index-1][_run][index] = atoi(tmpPlot->getId());
					(*plot_resp_observations)[constraint_index-1][_run][index] = tmpPlot->getResponseVariableAtIndex(0).getValue();
					index++;
				}
				else
					Logging::log_debug("PlotObserver::observe() index: ", Utility::int_to_char(index)," >  MaxPlots!",NULL);
			}
		}
	}

}

void PlotObserver::report_variable_names(ostream& strm) const
{
	if(number_of_constraints == 0)
	{
		Logging::log_debug("PlotObserver::report_variable_names() number_of_constraints is 0! Abort to avoid division by 0!",NULL);
		return;
	}

	if(!strcmp(plot_set_type, PS_PLOT_BUFFER))
	{
		for(int j=0; j<number_of_constraints; j++)
		{
			for(int i=1; i<=runs; i++)
			{
				if((*plot_pred_observations)[j][i-1][0] != NULL_FLOAT)
				{
					strm<<i<<"_"<<(*constraint_values)[j]<<"_ID,";
					strm<<i<<"_"<<(*constraint_values)[j]<<"_P,";
					strm<<i<<"_"<<(*constraint_values)[j]<<"_R,";
				}
			}
		}
	}
	else
	{
		for(int j=0; j<number_of_constraints; j++)
		{
			for(int i=1; i<=runs; i++)
			{
				if((*plot_pred_observations)[j][i-1][0] != NULL_FLOAT)
				{
					strm<<i<<"_"<<(*constraint_values)[j]<<"_ID,";
					strm<<i<<"_"<<(*constraint_values)[j]<<"_P,";
					strm<<i<<"_"<<(*constraint_values)[j]<<"_ID,";
					strm<<i<<"_"<<(*constraint_values)[j]<<"_R,";
				}
			}
		}
	}

	strm<<"\n";
}

void PlotObserver::report(ostream& strm) const
{
	int dot_gap = max_plots/(max_plots<10?max_plots:10);
	for(int k=0; k<max_plots; k++)
	{
		if(k==dot_gap)
		{
			dot_gap+=max_plots/(max_plots<10?max_plots:10);
			cout<<".";
		}

		if(!strcmp(plot_set_type, PS_PLOT_BUFFER))
		{
			for(int j=0; j<number_of_constraints; j++)
			{
				for(int i=0; i<runs; i++)
				{
					int sampled_plots = (*plot_pred_observations)[j][i].entries();

					double tmp_id = (*pred_plot_id_observations)[j][i][k];
					double tmp_x = (*plot_pred_observations)[j][i][k];
					double tmp_y = (*plot_resp_observations)[j][i][k];
					
					if(tmp_id == 0 && tmp_x == NULL_FLOAT && tmp_y == NULL_FLOAT)
					{
						if(sampled_plots != 0)
							strm<<",,,";
					}
					else
						strm<<tmp_id<<","<<tmp_x<<","<<tmp_y<<",";
				}
			}
		}
		else
		{
			for(int j=0; j<number_of_constraints; j++)
			{
				for(int i=0; i<runs; i++)
				{
					int sampled_plots = (*plot_pred_observations)[j][i].entries();

					double tmp_p_id = (*pred_plot_id_observations)[j][i][k];
					double tmp_r_id = (*resp_plot_id_observations)[j][i][k];
					double tmp_x = (*plot_pred_observations)[j][i][k];
					double tmp_y = (*plot_resp_observations)[j][i][k];
					
					if(tmp_p_id == 0 && tmp_r_id == 0 && tmp_x == NULL_FLOAT && tmp_y == NULL_FLOAT)
					{
						if(sampled_plots != 0)
							strm<<",,,,";
					}
					else
						strm<<tmp_p_id<<","<<tmp_x<<","<<tmp_r_id<<","<<tmp_y<<",";
				}
			}
		}

		strm<<"\n";
	}
}

#endif