/*FOCUS-2.0 is a sampling and statistical data analysis program
for spatial data sets. Copyright (C) 2003  Lutz Tischendorf

This program is free software; you can redistribute it
and/or modify it under the terms of the GNU General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public
License along with this program; if not, write to the Free
Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.*/

#ifndef __INPUTREADER_CPP
#define __INPUTREADER_CPP

#include "inputreader.h"
#include "focusutil.h"
#include "focusregistry.h"
#include "plot.h"
#include "distance.h"
#include "variable.h"
#include "defs.h"
#include <algorithm>
#include <set>

InputReader::InputReader(FocusRegistry*	_registry, const char* _filename)
{
	registry = _registry;
	registry->registerInputReader(this);

	Logging::log_debug("InputReader::InputReader(",_filename,") read configuration file...",NULL);

	xml_parameters = new TiXmlDocument(_filename);	
	if(!xml_parameters->LoadFile())
		throw FocusException("InputReader::InputReader could not read '", _filename, "'!");	

	Logging::log_debug("InputReader::InputReader(",_filename,") read configuration file DONE.",NULL);
}

InputReader::~InputReader()
{
}

void InputReader::readConfiguationFile(const char* _filename) throw (FocusException)
{
	Logging::log_debug("InputReader::readConfiguationFile(",_filename,")...",NULL);

	delete xml_parameters;

	xml_parameters = new TiXmlDocument(_filename);	
	if(!xml_parameters->LoadFile())
		throw FocusException("InputReader::readConfiguationFile() could not read '", _filename, "'!");	

	Logging::log_debug("InputReader::readConfiguationFile(",_filename,") DONE.",NULL);
}

const char* InputReader::getDistanceMatrixFileName() const throw (FocusException)
{
	Logging::log_debug("InputReader::getDistanceMatrixFileName()...",NULL);

	TiXmlElement* input_files = xml_parameters->FirstChildElement(TAG_NAME_FOCUS)
											  ->FirstChildElement(TAG_NAME_PLOTSET)
											  ->FirstChildElement(TAG_NAME_INPUTFILES)
											  ->FirstChildElement(TAG_NAME_PLOT_DISTANCES);
	if(!input_files)
		throw FocusException("InputReader::getPlotDistanceMatrixFileName()'",TAG_NAME_PLOT_DISTANCES,"' File not found!");
	
	TiXmlNode* filename = input_files->FirstChild();

	if(!filename)
		throw FocusException("InputReader::getPlotDistanceMatrixFileName() no filename for tag '",TAG_NAME_PLOT_DISTANCES,"'");

	const char* value_text = filename->Value().c_str();

	Logging::log_debug("InputReader::getDistanceMatrixFileName() DONE.",NULL);
	return value_text;
}

const char* InputReader::getPredictorVariablesFileName() const throw (FocusException)
{
	Logging::log_debug("InputReader::getPredictorVariablesFileName()...",NULL);

	TiXmlElement* input_files = xml_parameters->FirstChildElement(TAG_NAME_FOCUS)
											  ->FirstChildElement(TAG_NAME_PLOTSET)
											  ->FirstChildElement(TAG_NAME_INPUTFILES)
											  ->FirstChildElement(TAG_NAME_PREDICTOR_VARIABLES);
	if(!input_files)
		throw FocusException("InputReader::getPredictorVariablesFileName()'",TAG_NAME_PREDICTOR_VARIABLES,"' not found!");
	
	TiXmlNode* filename = input_files->FirstChild();

	if(!filename)
		throw FocusException("InputReader::getPredictorVariablesFileName() no filename for tag '",TAG_NAME_PREDICTOR_VARIABLES,"'");

	const char* value_text = filename->Value().c_str();
	Logging::log_debug("InputReader::getPredictorVariablesFileName() DONE.",NULL);

	return value_text;
}

const char* InputReader::getResponseVariablesFileName() const throw (FocusException)
{
	Logging::log_debug("InputReader::getResponseVariablesFileName()...",NULL);

	TiXmlElement* input_files = xml_parameters->FirstChildElement(TAG_NAME_FOCUS)
											  ->FirstChildElement(TAG_NAME_PLOTSET)
											  ->FirstChildElement(TAG_NAME_INPUTFILES)
											  ->FirstChildElement(TAG_NAME_RESPONSE_VARIABLES);
	if(!input_files)
		throw FocusException("InputReader::getResponseVariablesFileName()'",TAG_NAME_RESPONSE_VARIABLES,"' not found!");
	
	TiXmlNode* filename = input_files->FirstChild();

	if(!filename)
		throw FocusException("InputReader::getResponseVariablesFileName() no filename for tag '",TAG_NAME_RESPONSE_VARIABLES,"'");

	const char* value_text = filename->Value().c_str();

	Logging::log_debug("InputReader::getResponseVariablesFileName() DONE.",NULL);
	return value_text;
}

const char* InputReader::getOutputBaseName() const throw (FocusException)
{
	Logging::log_debug("InputReader::getOutputBaseName()...",NULL);

	TiXmlElement* output_files = xml_parameters->FirstChildElement(TAG_NAME_FOCUS)
											  ->FirstChildElement(TAG_NAME_PLOTSET)
											  ->FirstChildElement(TAG_NAME_OUTPUTFILES)
											  ->FirstChildElement(TAG_NAME_BASENAME);
	if(!output_files)
		throw FocusException("InputReader::getOutputBaseName()'",TAG_NAME_BASENAME,"' not found!");
	
	TiXmlNode* base_name = output_files->FirstChild();

	if(!base_name)
		throw FocusException("InputReader::getOutputBaseName() no filename for tag '",TAG_NAME_BASENAME,"'");

	const char* value_text = base_name->Value().c_str();
	Logging::log_debug("InputReader::getOutputBaseName() DONE.",NULL);
	return value_text;
}

bool InputReader::getTimeStamp() const throw (FocusException)
{
	Logging::log_debug("InputReader::getTimeStamp()...",NULL);

	TiXmlElement* time_stamp_el = xml_parameters->FirstChildElement(TAG_NAME_FOCUS)
											  ->FirstChildElement(TAG_NAME_PLOTSET)
											  ->FirstChildElement(TAG_NAME_OUTPUTFILES)
											  ->FirstChildElement(TAG_NAME_TIMESTAMP);
	if(!time_stamp_el)
		throw FocusException("InputReader::getOutputBaseName()'",TAG_NAME_TIMESTAMP,"' not found!");
	
	TiXmlNode* time_stamp_nod = time_stamp_el->FirstChild();

	if(!time_stamp_nod)
		throw FocusException("InputReader::getOutputBaseName() invalid value for tag '",TAG_NAME_TIMESTAMP,"'");

	const char* value_text = time_stamp_nod->Value().c_str();

	Logging::log_debug("InputReader::getTimeStamp() DONE.",NULL);

	if(!strcmp(value_text,"yes") || !strcmp(value_text, "YES"))
		return true;
	else if(!strcmp(value_text,"no") || !strcmp(value_text, "NO"))
		return false;
	else 
		throw FocusException("InputReader::getOutputBaseName() invalid value for tag '",TAG_NAME_TIMESTAMP,"'");
}

unsigned short InputReader::getNumberOfIterations() const throw (FocusException)
{
	Logging::log_debug("InputReader::getNumberOfIterations()...",NULL);

	TiXmlElement* configuration = xml_parameters->FirstChildElement("Focus")->FirstChildElement("Configuration")->FirstChildElement(TAG_NAME_NUMBER_OF_ITERATIONS);
	if(!configuration)
		throw FocusException("InputReader::getNumberOfIterations()'",TAG_NAME_NUMBER_OF_ITERATIONS,"' not found!");
	
	TiXmlNode* value = configuration->FirstChild();

	if(!value)
		throw FocusException("InputReader::getNumberOfIterations() no value for tag '",TAG_NAME_NUMBER_OF_ITERATIONS,"'");

	const char* value_text = value->Value().c_str();

	if(!isdigit(*value_text))
		throw FocusException("InputReader::getNumberOfIterations() invalid value for tag '",TAG_NAME_NUMBER_OF_ITERATIONS,"'");

	Logging::log_debug("InputReader::getNumberOfIterations() DONE.",NULL);

	return (unsigned short)atoi(value_text);
}

bool InputReader::getIncludeZeroRegression() const throw (FocusException)
{
	Logging::log_debug("InputReader::getIncludeZeroRegression()...",NULL);

	TiXmlElement* configuration = xml_parameters->FirstChildElement(TAG_NAME_FOCUS)
												->FirstChildElement(TAG_NAME_CONFIGURATION)
												->FirstChildElement(TAG_NAME_INCLUDE_ZERO_REGRESSION);
	if(!configuration)
		throw FocusException("InputReader::getIncludeZeroRegression()'",TAG_NAME_INCLUDE_ZERO_REGRESSION,"' not found!");
	
	TiXmlNode* value = configuration->FirstChild();

	if(!value)
		throw FocusException("InputReader::getIncludeZeroRegression() no value for tag '",TAG_NAME_INCLUDE_ZERO_REGRESSION,"'");

	const char* value_text = value->Value().c_str();

	Logging::log_debug("InputReader::getIncludeZeroRegression() DONE.",NULL);

	if(!strcmp(value_text,"yes") || !strcmp(value_text, "YES"))
		return true;

	return false;
}

unsigned short InputReader::getMaximumPlotsPerIteration() const throw (FocusException)
{
	Logging::log_debug("InputReader::getMaximumPlotsPerIteration()...",NULL);

	TiXmlElement* configuration = xml_parameters->FirstChildElement(TAG_NAME_FOCUS)
												->FirstChildElement(TAG_NAME_CONFIGURATION)
											    ->FirstChildElement(TAG_NAME_PLOTS_PER_ITERATION);
	if(!configuration)
		throw FocusException("InputReader::getMaximumPlotsPerIteration()'",TAG_NAME_PLOTS_PER_ITERATION,"' not found!");
	
	TiXmlNode* value = configuration->FirstChild();

	if(!value)
		throw FocusException("InputReader::getMaximumPlotsPerIteration() no value for tag '",TAG_NAME_PLOTS_PER_ITERATION,"'");

	const char* value_text = value->Value().c_str();

	if(!isdigit(*value_text))
		throw FocusException("InputReader::getMaximumPlotsPerIteration() invalid value for tag '",TAG_NAME_PLOTS_PER_ITERATION,"'");

	Logging::log_debug("InputReader::getMaximumPlotsPerIteration() DONE.",NULL);
	return (unsigned short)atoi(value_text);
}

unsigned int InputReader::getDuplicateRetrials() const throw (FocusException)
{
	Logging::log_debug("InputReader::getDuplicateRetrials()...",NULL);

	TiXmlElement* configuration = xml_parameters->FirstChildElement(TAG_NAME_FOCUS)
												->FirstChildElement(TAG_NAME_CONFIGURATION)
											    ->FirstChildElement(TAG_NAME_DUPLICATE_SAMPLES)
												->FirstChildElement(TAG_NAME_RETRIALS);
	if(!configuration)
		throw FocusException("InputReader::getDuplicateRetrials()'",TAG_NAME_RETRIALS,"' not found!");
	
	TiXmlNode* value = configuration->FirstChild();

	if(!value)
		throw FocusException("InputReader::getDuplicateRetrials() no value for tag '",TAG_NAME_RETRIALS,"'");

	const char* value_text = value->Value().c_str();

	if(!isdigit(*value_text))
		throw FocusException("InputReader::getDuplicateRetrials() invalid value for tag '",TAG_NAME_RETRIALS,"'");

	Logging::log_debug("InputReader::getDuplicateRetrials() DONE.",NULL);
	return (unsigned int)atoi(value_text);
}

bool InputReader::getDuplicatesAllowedFlag() const throw (FocusException)
{
	Logging::log_debug("InputReader::getDuplicatesAllowedFlag()...",NULL);

	TiXmlElement* configuration = xml_parameters->FirstChildElement(TAG_NAME_FOCUS)
												->FirstChildElement(TAG_NAME_CONFIGURATION)
											    ->FirstChildElement(TAG_NAME_DUPLICATE_SAMPLES)
												->FirstChildElement(TAG_NAME_ALLOW);
	if(!configuration)
		throw FocusException("InputReader::getDuplicatesAllowedFlag()'",TAG_NAME_ALLOW,"' not found!");
	
	TiXmlNode* value = configuration->FirstChild();

	if(!value)
		throw FocusException("InputReader::getDuplicatesAllowedFlag() no value for tag '",TAG_NAME_ALLOW,"'");

	const char* value_text = value->Value().c_str();

	Logging::log_debug("InputReader::getDuplicatesAllowedFlag() DONE.",NULL);
	if(!strcmp(value_text,"yes") || !strcmp(value_text, "YES"))
		return true;

	return false;
}

bool InputReader::getUniqueEnforceFlag() const throw (FocusException)
{
	Logging::log_debug("InputReader::getUniqueEnforceFlag()...",NULL);

	TiXmlElement* configuration = xml_parameters->FirstChildElement(TAG_NAME_FOCUS)
												->FirstChildElement(TAG_NAME_CONFIGURATION)
											    ->FirstChildElement(TAG_NAME_UNIQUE_SAMPLES)
												->FirstChildElement(TAG_NAME_ENFORCE);
	if(!configuration)
		throw FocusException("InputReader::getUniqueEnforceFlag()'",TAG_NAME_ENFORCE,"' not found!");
	
	TiXmlNode* value = configuration->FirstChild();

	if(!value)
		throw FocusException("InputReader::getUniqueEnforceFlag() no value for tag '",TAG_NAME_ENFORCE,"'");

	const char* value_text = value->Value().c_str();

	Logging::log_debug("InputReader::getUniqueEnforceFlag() DONE.",NULL);
	if(!strcmp(value_text,"yes") || !strcmp(value_text, "YES"))
		return true;

	return false;
}

unsigned short InputReader::getMaximumOrder() const throw (FocusException)
{
	Logging::log_debug("InputReader::getMaximumOrder()...",NULL);

	TiXmlElement* maximum_order = xml_parameters->FirstChildElement(TAG_NAME_FOCUS)
												->FirstChildElement(TAG_NAME_CONFIGURATION)
											    ->FirstChildElement(TAG_NAME_FUNCTIONAL_STABILITY)
												->FirstChildElement(TAG_NAME_MAXIMUM_ORDER);
	if(!maximum_order)
		throw FocusException("InputReader::getMaximumOrder()'",TAG_NAME_MAXIMUM_ORDER,"' not found!");
	
	TiXmlNode* value = maximum_order->FirstChild();

	if(!value)
		throw FocusException("InputReader::getMaximumOrder() no value for tag '",TAG_NAME_MAXIMUM_ORDER,"'");

	const char* value_text = value->Value().c_str();

	if(!isdigit(*value_text) || (atoi(value_text) < 1 || atoi(value_text) > 5))
		throw FocusException("InputReader::getMaximumOrder() invalid value for tag '",TAG_NAME_MAXIMUM_ORDER,"'");

	Logging::log_debug("InputReader::getMaximumOrder() DONE.",NULL);
	return (unsigned short)atoi(value_text);
}

unsigned short InputReader::getPercentage() const throw (FocusException)
{
	Logging::log_debug("InputReader::getPercentage()...",NULL);

	TiXmlElement* percentage = xml_parameters->FirstChildElement(TAG_NAME_FOCUS)
											 ->FirstChildElement(TAG_NAME_CONFIGURATION)
											 ->FirstChildElement(TAG_NAME_FUNCTIONAL_STABILITY)
											 ->FirstChildElement(TAG_NAME_PERCENTAGE_OF_RUNS);
	if(!percentage)
		throw FocusException("InputReader::getPercentage()'",TAG_NAME_PERCENTAGE_OF_RUNS,"' not found!");
	
	TiXmlNode* value = percentage->FirstChild();

	if(!value)
		throw FocusException("InputReader::getPercentage() no value for tag '",TAG_NAME_PERCENTAGE_OF_RUNS,"'");

	const char* value_text = value->Value().c_str();

	if(!isdigit(*value_text) || (atoi(value_text) < 1 || atoi(value_text) > 100))
		throw FocusException("InputReader::getPercentage() invalid value for tag '",TAG_NAME_PERCENTAGE_OF_RUNS,"'");

	Logging::log_debug("InputReader::getPercentage() DONE.",NULL);
	return (unsigned short)atoi(value_text);
}

bool InputReader::getFunctionalStability() const throw (FocusException)
{
	Logging::log_debug("InputReader::getFunctionalStability()...",NULL);

	TiXmlElement* enable = xml_parameters->FirstChildElement(TAG_NAME_FOCUS)
												->FirstChildElement(TAG_NAME_CONFIGURATION)
											    ->FirstChildElement(TAG_NAME_FUNCTIONAL_STABILITY)
												->FirstChildElement(TAG_NAME_ENABLE);
	if(!enable)
		throw FocusException("InputReader::getFunctionalStability()'",TAG_NAME_ENABLE,"' not found!");
	
	TiXmlNode* value = enable->FirstChild();

	if(!value)
		throw FocusException("InputReader::getFunctionalStability() no value for tag '",TAG_NAME_ENABLE,"'");

	const char* value_text = value->Value().c_str();

	Logging::log_debug("InputReader::getFunctionalStability() DONE.",NULL);

	if(!strcmp(value_text,"yes") || !strcmp(value_text, "YES"))
		return true;

	return false;
}

bool InputReader::getStatistics() const throw (FocusException)
{
	Logging::log_debug("InputReader::getStatistics()...",NULL);

	TiXmlElement* statistics = xml_parameters->FirstChildElement(TAG_NAME_FOCUS)
											 ->FirstChildElement(TAG_NAME_CONFIGURATION)
											 ->FirstChildElement(TAG_NAME_STATISTICS);
	if(!statistics)
		throw FocusException("InputReader::getStatistics()'",TAG_NAME_STATISTICS,"' not found!");
	
	TiXmlNode* value = statistics->FirstChild();

	if(!value)
		throw FocusException("InputReader::getStatistics() no value for tag '",TAG_NAME_STATISTICS,"'");

	const char* value_text = value->Value().c_str();

	Logging::log_debug("InputReader::getStatistics() DONE.",NULL);

	if(!strcmp(value_text,"yes") || !strcmp(value_text, "YES"))
		return true;

	return false;
}

bool InputReader::getWritePlots() const throw (FocusException)
{
	Logging::log_debug("InputReader::getWritePlots()...",NULL);

	TiXmlElement* writeplots = xml_parameters->FirstChildElement(TAG_NAME_FOCUS)
											 ->FirstChildElement(TAG_NAME_CONFIGURATION)
											 ->FirstChildElement(TAG_NAME_WRITE_PLOTS);
	if(!writeplots)
		throw FocusException("InputReader::getWritePlots()'",TAG_NAME_WRITE_PLOTS,"' not found!");
	
	TiXmlNode* value = writeplots->FirstChild();

	if(!value)
		throw FocusException("InputReader::getWritePlots() no value for tag '",TAG_NAME_WRITE_PLOTS,"'");

	const char* value_text = value->Value().c_str();

	Logging::log_debug("InputReader::getWritePlots() DONE.",NULL);

	if(!strcmp(value_text,"yes") || !strcmp(value_text, "YES"))
		return true;

	return false;
}

plotset_types InputReader::getPlotSetType() const throw (FocusException)
{
	Logging::log_debug("InputReader::getPlotSetType()...",NULL);

	TiXmlElement* type = xml_parameters->FirstChildElement(TAG_NAME_FOCUS)
											  ->FirstChildElement(TAG_NAME_PLOTSET)
											  ->FirstChildElement(TAG_NAME_TYPE);
	if(!type)
		throw FocusException("InputReader::getPlotSetType()'",TAG_NAME_TYPE,"' not found!");
	
	TiXmlNode* value = type->FirstChild();

	if(!value)
		throw FocusException("InputReader::getPlotSetType() no value for tag '",TAG_NAME_TYPE,"'");

	const char* value_text = value->Value().c_str();

	Logging::log_debug("InputReader::getPlotSetType() DONE.",NULL);

	if(!strcmp(value_text,"Plot-Buffer") || !strcmp(value_text, "PLOT-BUFFER") || !strcmp(value_text, "plot-buffer"))
		return PLOT_BUFFER;
	
	if(!strcmp(value_text,"Plot-Plot") || !strcmp(value_text, "PLOT-PLOT") || !strcmp(value_text, "plot-plot"))
		return PLOT_PLOT;
	
	throw FocusException("InputReader::getPlotSetType() invalid value for tag '",TAG_NAME_TYPE,"'");
}
	
ConstraintList* InputReader::getConstraints() const throw (FocusException)
{
	Logging::log_debug("InputReader::getConstraints()...",NULL);

	TiXmlElement* constraint = xml_parameters->FirstChildElement(TAG_NAME_FOCUS)
											 ->FirstChildElement(TAG_NAME_PLOTSET)
											 ->FirstChildElement(TAG_NAME_CONSTRAINTS);
	if(!constraint)
		throw FocusException("InputReader::getConstraints()'",TAG_NAME_CONSTRAINTS,"' not found!");

	ConstraintList* constraints = new ConstraintList();
	const char* multiplier_text;
	const char* sample_type_text;
	double multiplier;
	sample_types sample_type;
	
	TiXmlNode* multiplier_node = constraint->FirstChildElement(TAG_NAME_MULTIPLIER)->FirstChild();

	if(!multiplier_node)
		throw FocusException("InputReader::getConstraints() no value for tag '", TAG_NAME_MULTIPLIER);

	multiplier_text = multiplier_node->Value().c_str();

	if(!isdigit(*multiplier_text))
		throw FocusException("InputReader::getConstraints() invalid value for tag '",TAG_NAME_MULTIPLIER,"'");

	multiplier = atof(multiplier_text);

	TiXmlNode* sample_type_node = constraint->FirstChildElement(TAG_NAME_SAMPLETYPE)->FirstChild();

	if(!sample_type_node)
		throw FocusException("InputReader::getConstraints() no value for tag '",TAG_NAME_SAMPLETYPE,"'");

	sample_type_text = sample_type_node->Value().c_str();

	if(!strcmp(sample_type_text,"outside") || !strcmp(sample_type_text, "OUTSIDE"))
		sample_type = OUTSIDE;
	else if(!strcmp(sample_type_text,"inside") || !strcmp(sample_type_text, "INSIDE"))
		sample_type = INSIDE;
	else
		throw FocusException("InputReader::getConstraints() invalid value for tag '",TAG_NAME_SAMPLETYPE,"'");

	TiXmlElement* value = constraint->FirstChildElement(TAG_NAME_DISTANCE)->FirstChildElement();
	while(value)
	{
		TiXmlNode* value_content = value->FirstChild();
		if(!value_content)
			throw FocusException("InputReader::getConstraints() no value for Distance constraint");

		const char* value_text = value_content->Value().c_str();
		if(!isdigit(*value_text))
			Logging::log_error("InputReader::getConstraints() value ",value_text, " not valid for Distance constraint!",NULL);
		else
			constraints->addConstraint(new DistanceConstraint("DISTANCE", atof(value_text), multiplier, sample_type));
		
		value = value->NextSiblingElement();
	}

	if(constraints->size() == 0)
		throw FocusException("InputReader::getConstraints() no values for '",TAG_NAME_DISTANCE,"'");
	
	Logging::log_debug("InputReader::getConstraints() DONE.",NULL);


	return constraints;
}

void InputReader::readPlotVariables(const char* _filename, variable_types _varType) throw (FocusException)
{
	Logging::log_debug("InputReader::readPlotVariables(",_filename,",",Utility::int_to_char(_varType),") ...",NULL);

	create_plots_and_variables(_filename, _varType);
	assign_distances(_varType);

	Logging::log_debug("InputReader::readPlotVariables(",_filename,") DONE.",NULL);
}

void InputReader::readPlotDistanceMatrix(const char* _filename) throw (FocusException)
{
	Logging::log_debug("InputReader::readPlotDistanceMatrix(",_filename,") ...",NULL);

	verify_distance_matrix_symmetry(_filename);
	create_plots_from_distance_matrix(_filename);
	assign_distances_from_distance_matrix(_filename);

	Logging::log_debug("InputReader::readPlotDistanceMatrix(",_filename,") DONE.",NULL);
}

void InputReader::readVariables(const char* _filename, variable_types _varType, categories _category) throw (FocusException)
{
	Logging::log_debug("InputReader::readVariables(",_filename,",",Utility::int_to_char(_varType),
						",",Utility::int_to_char(_category),") ...",NULL);

	if(_category == WITH_CONTEXT)
		create_cirular_plot_buffers(_filename);

	create_variables(_filename, _varType, _category);

	Logging::log_debug("InputReader::readVariables(",_filename,") DONE.",NULL);
}

int InputReader::read_number_of_lines_from_file(const char* _filename) const throw (FocusException)
{
	int lines=0;
	ifstream is(_filename, ios::in);
	if(!is)
		throw FocusException("InputReader::read_number_of_lines_from_file() ",
							   "cannot open file '", _filename, "'.");
	char ch;
	while(is.get(ch))
	if(ch == '\n')
		lines++;
	
	is.close();
	return lines;
}

void InputReader::verify_distance_matrix_symmetry(const char* _filename) const throw (FocusException)
{
	Logging::log_debug("InputReader::verify_distance_matrix_symmetry(",_filename,") ...",NULL);

	ifstream is(_filename, ios::in);
	if(is == NULL)
		throw FocusException("InputReader::verify_distance_matrix_symmetry() ", 
							   "cannot open file '", _filename, "'.");

	int xd=0;
	int yd=0;
	char ch;
	int count=0;

	while(is.get(ch))
    {
		if(ch != '\n' && !yd)
		{
			count++;
			if(ch != ';' && ch != ',' && ch != ' ')
			{
				if(count == 1)
					xd++;
			}
			else
				count = 0;
		}
		else if(ch == '\n')
			yd++;
    }
	is.close();

	if(xd != yd) 
		throw FocusException("InputReader::verify_distance_matrix_symmetry() distance matrix ",
							  "is not symmetrical!");
	
	Logging::log_debug("InputReader::verify_distance_matrix_symmetry(",_filename,") DONE.",NULL);
}

void InputReader::create_plots_from_distance_matrix(const char* _filename) const throw (FocusException)
{
	Logging::log_debug("InputReader::create_plots_from_distance_matrix(",_filename,") ...",NULL);

	ifstream is(_filename, ios::in);
	if(is == NULL)
		throw FocusException("InputReader::create_plots_from_distance_matrix() cannot open file '", _filename, "'.");

	PlotList* plots = new PlotList();
	PlotSet* plotSet = new PlotSet(plots, NULL, WITH_CONTEXT, PS_PLOT_BUFFER);
	PlotSetList* plotSets = NULL;
	if(!registry->getPlotSetList())
		plotSets = new PlotSetList(registry);

	plotSets->addPlotSet(plotSet);
	
	char line[MAX_PLOTS];
	char line_buffer[MAX_PLOTS];
	char* delims={";, "};
	char* plot_id;

	is.getline(line,MAX_PLOTS,'\n');
	sprintf(line_buffer,"%s",line);

	plot_id=strtok(line_buffer,delims);
	plot_id=strtok(NULL,delims);	//skip first column

	while(plot_id)
	{
		plots->addPlot(new Plot(plot_id));
		plot_id=strtok(NULL,delims);
	}
	
	is.close();

	Logging::log_debug("InputReader::create_plots_from_distance_matrix(",_filename,") DONE.",NULL);
}

void InputReader::assign_distances_from_distance_matrix(const char* _filename) throw (FocusException)
{
	Logging::log_debug("InputReader::assign_distances(",_filename,") ...",NULL);

	ifstream is(_filename, ios::in);
	if(is == NULL)
		throw FocusException("InputReader::assign_distances() ", 
							   "cannot open file '", _filename, "'.");

	char buf[MAX_PLOTS];
	char* delims={";, "};
	char* distance;
	unsigned int plot_index = 0;

	DistanceList* distances = new DistanceList();
	PlotSetList* plotSets = NULL;

	if(!registry->getPlotSetList())
		plotSets = new PlotSetList(registry);
	else
		plotSets = registry->getPlotSetList();

	PlotSet* plotSet = plotSets->getPlotSetForReferenceName(PS_PLOT_BUFFER);
	if(!plotSet)
	{
		plotSet = new PlotSet(NULL, distances, WITH_CONTEXT, PS_PLOT_BUFFER);
		plotSets->addPlotSet(plotSet);
	}
	else
		plotSet->setDistanceList(distances);
	
	PlotList::iterator first_plot_iterator = plotSet->getPlotList()->begin();
	PlotList::iterator second_plot_iterator = plotSet->getPlotList()->begin();

	is.getline(buf,MAX_PLOTS,'\n');			//skip first line

	while(first_plot_iterator != plotSet->getPlotList()->end())
	{
		second_plot_iterator = plotSet->getPlotList()->begin();

		is.getline(buf,MAX_PLOTS,'\n');
		distance = strtok(buf,delims);
		distance = strtok(NULL,delims);
		plot_index++;
	
		for(int i=0; i<plot_index; i++)
		{
			second_plot_iterator++;
			distance = strtok(NULL,delims);
		}

		while(second_plot_iterator != plotSet->getPlotList()->end())
		{
			if(distance && isdigit(*distance))
			{
				distances->addDistance(*(new Distance(atof(distance), 
									   (*first_plot_iterator),
									   (*second_plot_iterator))));
			}
			else
				Logging::log_error("InputReader::assign_distances(",_filename,") malformed distance value: ", distance, NULL);
			
			distance = strtok(NULL,delims);
			second_plot_iterator++;
		}
		first_plot_iterator++;
	}

	is.close();
	Logging::log_debug("InputReader::assign_distances(",_filename,") DONE.",NULL);
}

void InputReader::create_cirular_plot_buffers(const char* _filename) const throw (FocusException)
{
	Logging::log_debug("InputReader::create_cirular_plot_buffers(",_filename,") ...",NULL);

	ifstream is(_filename, ios::in);
	if(is == NULL)
		throw FocusException("InputReader::create_cirular_plot_buffers() cannot open file '", _filename, "'.");
	
	SpatialContextList* cirularBuffers = new SpatialContextList(registry);

	char line[MAX_CONSTRAINTS];
	char line_buffer[MAX_CONSTRAINTS];
	char* delims={";, "};
	char* buffer_value;

	is.getline(line,MAX_CONSTRAINTS,'\n');
	sprintf(line_buffer,"%s",line);

	buffer_value=strtok(line_buffer,delims);
	buffer_value=strtok(NULL,delims);	//skip first column

	while(buffer_value)
	{
		if(isdigit(*buffer_value))
			cirularBuffers->addSpatialContext(new CircularPlotBuffer("buffer",atof(buffer_value)));
		else
			Logging::log_error("InputReader::create_cirular_plot_buffers(",_filename,") malformed buffer value: ", buffer_value, NULL);
		buffer_value=strtok(NULL,delims);
	}
	
	is.close();

	Logging::log_debug("InputReader::create_cirular_plot_buffers(",_filename,") DONE.",NULL);
}

void InputReader::create_variables(const char* _filename, variable_types _varType, categories _category) const throw (FocusException)
{
	Logging::log_debug("InputReader::create_variables(",_filename,") ...",NULL);

	ifstream is(_filename, ios::in);
	if(is == NULL)
		throw FocusException("InputReader::create_variables() cannot open file '", _filename, "'.");

	char buf[MAX_PLOTS];
	char* delims={";, "};
	char* value;
	char* plot_id;
	SpatialContextList::iterator context_iterator = registry->getSpatialContextList()->begin();

	int lines = read_number_of_lines_from_file(_filename);

	if(_varType == PREDICTOR && _category == WITH_CONTEXT)
		is.getline(buf,MAX_PLOTS,'\n');			//skip first header line
	
	for(int i=0; i<lines; i++)
	{
		is.getline(buf,MAX_PLOTS,'\n');
		plot_id = strtok(buf,delims);

		if(plot_id)
		{
			Plot* tmpPlot = registry->getPlotSetList()->getPlotSetForReferenceName(PS_PLOT_BUFFER)->getPlotList()->findPlot(plot_id);
			value = strtok(NULL,delims);

			if(_category == WITH_CONTEXT)
				context_iterator = registry->getSpatialContextList()->begin();
			
			while(value && tmpPlot)
			{
				if(isdigit(*value))
				{
					if(_varType == PREDICTOR && _category == WITH_CONTEXT)
					{
						Variable* var = new Variable("predictor", atof(value), *context_iterator);
						tmpPlot->addPredictor_variable(*var);
					}
					else
					{
						Variable* var = new Variable("response", atof(value));
						tmpPlot->addResponse_variable(*var);
					}
				}
				else
					Logging::log_error("InputReader::create_variables(",_filename,") malformed value: ", value, NULL);

				value = strtok(NULL,delims);
				
				if(_category == WITH_CONTEXT)
					context_iterator++;
			}
		}
	}

	is.close();

	Logging::log_debug("InputReader::create_variables(",_filename,") DONE.",NULL);
}

void InputReader::create_plots_and_variables(const char* _filename, variable_types _varType) const throw (FocusException)
{
	Logging::log_debug("InputReader::create_plots_and_variables(",_filename,",",Utility::int_to_char(_varType),") ...",NULL);

	ifstream is(_filename, ios::in);
	if(is == NULL)
		throw FocusException("InputReader::create_plots_and_variables() cannot open file '", _filename, "'.");

	const char* ps_type = (_varType == PREDICTOR ? PS_PLOT_PLOT_PREDICTOR : PS_PLOT_PLOT_RESPONSE);
	PlotList* plots = new PlotList();
	PlotSet* plotSet = new PlotSet(plots, NULL, WITHOUT_CONTEXT, ps_type);
	PlotSetList* plotSets = registry->getPlotSetList();	
	if(!plotSets)
		plotSets = new PlotSetList(registry);

	plotSets->addPlotSet(plotSet);

	char line[MAX_PLOTS];
	char* delims={";, "};
	char* plot_id;
	char* value;
	char* x_coord;
	char* y_coord;

	int lines = read_number_of_lines_from_file(_filename);
	
	for(int i=0; i<lines; i++)
	{
		is.getline(line,MAX_PLOTS,'\n');

		plot_id = strtok(line,delims);
		x_coord = strtok(NULL,delims);
		y_coord = strtok(NULL,delims);

		if(plot_id && x_coord && y_coord && isdigit(*x_coord) && isdigit(*y_coord))
		{
			Plot* tmpPlot = new Plot(plot_id, atof(x_coord), atof(y_coord));
			plots->addPlot(tmpPlot);
			
			value = strtok(NULL,delims);
			
			while(value && tmpPlot)
			{
				if(isdigit(*value))
				{
					if(_varType == PREDICTOR)
					{
						Variable* var = new Variable("predictor", atof(value));
						tmpPlot->addPredictor_variable(*var);
					}
					else
					{
						Variable* var = new Variable("response", atof(value));
						tmpPlot->addResponse_variable(*var);
					}
				}
				else
					Logging::log_error("InputReader::create_plots_and_variables(",_filename,") malformed value: ", value, NULL);

				value = strtok(NULL,delims);
			}
		}
	}
	is.close();

	Logging::log_debug("InputReader::create_plots_and_variables(",_filename,") DONE.",NULL);
}

void InputReader::assign_distances(variable_types _varType) throw (FocusException)
{
	Logging::log_debug("InputReader::assign_distances(",Utility::int_to_char(_varType),") ...",NULL);

	const char* ps_type = (_varType == PREDICTOR ? PS_PLOT_PLOT_PREDICTOR : PS_PLOT_PLOT_RESPONSE);
	unsigned int plot_index = 0;
	DistanceList* distances = new DistanceList();
	PlotSetList* plotSets = registry->getPlotSetList();

	if(!plotSets)
		throw FocusException("InputReader::calculate_plot_distances() cannot find PlotSetList.");

	PlotSet* plotSet = plotSets->getPlotSetForReferenceName(ps_type);

	if(!plotSet)
		throw FocusException("InputReader::calculate_plot_distances() cannot find PlotSet for ", ps_type, " variables.");	
	
	plotSet->setDistanceList(distances);
	
	PlotList::iterator first_plot_iterator = plotSet->getPlotList()->begin();
	PlotList::iterator second_plot_iterator = plotSet->getPlotList()->begin();

	while(first_plot_iterator != plotSet->getPlotList()->end())
	{
		second_plot_iterator = plotSet->getPlotList()->begin();

		plot_index++;

		for(int i=0; i<plot_index; i++)
			second_plot_iterator++;

		while(second_plot_iterator != plotSet->getPlotList()->end())
		{
			distances->addDistance(*(new Distance((*first_plot_iterator),(*second_plot_iterator))));
			second_plot_iterator++;
		}
		first_plot_iterator++;
	}

	distances->calculateDistances();
	
	Logging::log_debug("InputReader::assign_distances() DONE.",NULL);
}


void InputReader::validatePlotBufferDataSet(const char* _matrixfilename, const char* _predictorfilename, const char* _responsefilename) throw (FocusException)
{
	Logging::log_debug("InputReader::validatePlotBufferDataSet(",_matrixfilename,",",_predictorfilename,",",_responsefilename,") ...",NULL);

	ifstream is(_matrixfilename, ios::in);
	if(is == NULL)
		throw FocusException("InputReader::validateDataSet() cannot open file '", _matrixfilename, "'.");

	set<string, ltstr> plotids;
	set<string, ltstr> predplotids;
	set<string, ltstr> respplotids;

	char line[MAX_PLOTS];
	char line_buffer[MAX_PLOTS];
	char* delims={";, "};
	char* plot_id;

	//read plotids for matrix columns (first row)
	is.getline(line,MAX_PLOTS,'\n');
	sprintf(line_buffer,"%s",line);
	plot_id=strtok(line_buffer,delims);
	plot_id=strtok(NULL,delims);	//skip first column
	while(plot_id)
	{
		plotids.insert(string(plot_id));
		plot_id=strtok(NULL,delims);
	}

	//check plotids from matrix rows agains columns (first column of each row)
	int lines = read_number_of_lines_from_file(_matrixfilename);
	for(int i=0; i<lines; i++)
	{
		is.getline(line,MAX_PLOTS,'\n');
		sprintf(line_buffer,"%s",line);
		plot_id=strtok(line_buffer,delims);
		if(plot_id)
		{	
			if(plotids.find(string(plot_id)) == plotids.end())
				throw FocusException("InputReader::validateDataSet() the plotID '",plot_id, "' in the distance matrix seems wrong!");

		}
	}
	is.close();

	//check predictor plotids against matrix plotids
	ifstream is2(_predictorfilename, ios::in);
	if(is2 == NULL)
		throw FocusException("InputReader::validateDataSet() cannot open file '", _predictorfilename, "'.");
	lines = read_number_of_lines_from_file(_predictorfilename);
	for(i=0; i<lines; i++)
	{
		
		is2.getline(line,MAX_PLOTS,'\n');
		if(i>0) //skip first line
		{
			sprintf(line_buffer,"%s",line);
			plot_id=strtok(line_buffer,delims);
			if(plot_id)
			{	
				if(plotids.find(string(plot_id)) == plotids.end())
					throw FocusException("InputReader::validateDataSet(): the plotID '",plot_id, "' in the predicor variables seems wrong!");
				else
					predplotids.insert(plot_id);
			}
		}
	}
	if(predplotids.size() != plotids.size())
		throw FocusException("InputReader::validatePlotBufferDataSet(): # of predicor variables does not match # of plots in distance matrix!");
					
	is2.close();

	//check response plotids against matrix plotids
	ifstream is3(_responsefilename, ios::in);
	if(is3 == NULL)
		throw FocusException("InputReader::validateDataSet() cannot open file '", _responsefilename, "'.");
	lines = read_number_of_lines_from_file(_responsefilename);
	for(i=0; i<lines; i++)
	{
		is3.getline(line,MAX_PLOTS,'\n');
		sprintf(line_buffer,"%s",line);
		plot_id=strtok(line_buffer,delims);
		if(plot_id)
		{	
			if(plotids.find(string(plot_id)) == plotids.end())
				throw FocusException("InputReader::validateDataSet() the plotID '",plot_id, "' in the response variables seems wrong!");
			else
				respplotids.insert(plot_id);
		}
	}
	if(respplotids.size() != plotids.size())
		throw FocusException("InputReader::validateDataSet(): # of response variables does not match # of plots in distance matrix!");
	
	is3.close();
	Logging::log_debug("InputReader::validateDataSet(",_matrixfilename,",",_predictorfilename,",",_responsefilename,") DONE.",NULL);
}

void InputReader::validatePlotPlotDataSet(const char* _predictorfilename, const char* _responsefilename) throw (FocusException)
{
	Logging::log_debug("InputReader::validatePlotPlotDataSet(",_predictorfilename,",",_responsefilename,") ...",NULL);

	set<string, ltstr> predplotids;
	set<string, ltstr> respplotids;

	char line[MAX_PLOTS];
	char line_buffer[MAX_PLOTS];
	char* delims={";, "};
	char* plot_id;
	int lines = 0;

	//read predictor plots
	ifstream is2(_predictorfilename, ios::in);
	if(is2 == NULL)
		throw FocusException("InputReader::validateDataSet() cannot open file '", _predictorfilename, "'.");
	lines = read_number_of_lines_from_file(_predictorfilename);
	for(int i=0; i<lines; i++)
	{
		
		is2.getline(line,MAX_PLOTS,'\n');
		if(i>0) //skip first line
		{
			sprintf(line_buffer,"%s",line);
			plot_id=strtok(line_buffer,delims);
			if(plot_id)
				predplotids.insert(string(plot_id));
		}
	}
	is2.close();

	//check response plotids against predictor plotids
	ifstream is3(_responsefilename, ios::in);
	if(is3 == NULL)
		throw FocusException("InputReader::validateDataSet() cannot open file '", _responsefilename, "'.");
	lines = read_number_of_lines_from_file(_responsefilename);
	for(i=0; i<lines; i++)
	{
		is3.getline(line,MAX_PLOTS,'\n');
		if(i>0)
		{
			sprintf(line_buffer,"%s",line);
			plot_id=strtok(line_buffer,delims);
			if(plot_id)
			{	
				if(predplotids.find(string(plot_id)) == predplotids.end())
					throw FocusException("InputReader::validateDataSet() the plotID '",plot_id, "' in the response variables seems wrong!");
				else
					respplotids.insert(plot_id);
			}
		}
	}
	if(respplotids.size() != predplotids.size())
		throw FocusException("InputReader::validateDataSet(): # of response variables does not match # of predictor variables!");
	
	is3.close();
	Logging::log_debug("InputReader::validatePlotPlotDataSet(",_predictorfilename,",",_responsefilename,") DONE.",NULL);
}


const char* InputReader::TAG_NAME_FOCUS	=					"Focus";
const char* InputReader::TAG_NAME_CONFIGURATION	=			"Configuration";
const char* InputReader::TAG_NAME_NUMBER_OF_ITERATIONS =	"Number_Of_Iterations";
const char* InputReader::TAG_NAME_PLOTS_PER_ITERATION =		"Maximum_Plots_Per_Iteration";
const char* InputReader::TAG_NAME_INCLUDE_ZERO_REGRESSION =	"Report_Zero_Regressions";
const char* InputReader::TAG_NAME_DUPLICATE_SAMPLES =		"DuplicateSamples";
const char* InputReader::TAG_NAME_ALLOW =					"Allow";
const char* InputReader::TAG_NAME_RETRIALS =				"Retrials";
const char* InputReader::TAG_NAME_UNIQUE_SAMPLES =			"UniqueSamples";
const char* InputReader::TAG_NAME_ENFORCE =					"Enforce";
const char* InputReader::TAG_NAME_FUNCTIONAL_STABILITY =	"FunctionalStability";
const char* InputReader::TAG_NAME_ENABLE =					"Enable";
const char* InputReader::TAG_NAME_MAXIMUM_ORDER =			"Maximum_Order";
const char* InputReader::TAG_NAME_PERCENTAGE_OF_RUNS =		"Percentage_Of_Runs";
const char* InputReader::TAG_NAME_STATISTICS =				"Statistics";
const char* InputReader::TAG_NAME_PLOTSET =					"PlotSet";
const char* InputReader::TAG_NAME_TYPE =					"Type";
const char* InputReader::TAG_NAME_INPUTFILES =				"InputFiles";
const char* InputReader::TAG_NAME_PLOT_DISTANCES =			"Plot_Distances";
const char* InputReader::TAG_NAME_PREDICTOR_VARIABLES =		"Predictor_Variables";
const char* InputReader::TAG_NAME_RESPONSE_VARIABLES =		"Response_Variables";
const char* InputReader::TAG_NAME_OUTPUTFILES =				"OutputFiles";
const char* InputReader::TAG_NAME_BASENAME =				"BaseName";
const char* InputReader::TAG_NAME_CONSTRAINTS =				"Constraints";
const char* InputReader::TAG_NAME_DISTANCE =				"Distance";
const char* InputReader::TAG_NAME_VALUE =					"Value";
const char* InputReader::TAG_NAME_MULTIPLIER =				"Multiplier";
const char* InputReader::TAG_NAME_SAMPLETYPE =				"SampleType";
const char* InputReader::TAG_NAME_TIMESTAMP =				"TimeStamp";
const char* InputReader::TAG_NAME_WRITE_PLOTS =				"WritePlots";

#endif
