/*FOCUS-2.0 is a sampling and statistical data analysis program
for spatial data sets. Copyright (C) 2003  Lutz Tischendorf

This program is free software; you can redistribute it
and/or modify it under the terms of the GNU General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public
License along with this program; if not, write to the Free
Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.*/

#ifndef __STATISTICS_CPP
#define __STATISTICS_CPP

#include "statistics.h"
#include "focusregistry.h"
#include "focusutil.h"
#include "logging.h"

#include <gsl/gsl_fit.h>
#include <gsl/gsl_statistics.h>
#include <gsl/gsl_multifit.h>

AnovaData::AnovaData()
{
	sst = NULL_DOUBLE;
	ssr = NULL_DOUBLE;
	sse = NULL_DOUBLE;
	mse = NULL_DOUBLE;
	f =	  NULL_DOUBLE;
	rr =  NULL_DOUBLE;
	r =	  NULL_DOUBLE;
	n  =  0;
}
	
void AnovaData::report(ostream& strm) const
{
	strm<<"SST="<<(sst == NULL_DOUBLE ? "INVALID" : Utility::float_to_char(sst));
	strm<<" SSR="<<(ssr == NULL_DOUBLE ? "INVALID" : Utility::float_to_char(ssr));
	strm<<" SSE="<<(sse == NULL_DOUBLE ? "INVALID" : Utility::float_to_char(sse));
	strm<<" MSE="<<(mse == NULL_DOUBLE ? "INVALID" : Utility::float_to_char(mse));
	strm<<" F="<<(f == NULL_DOUBLE ? "INVALID" : Utility::float_to_char(f));
	strm<<" R-Square="<<(rr == NULL_DOUBLE ? "INVALID" : Utility::float_to_char(rr));
	strm<<" r="<<(rr == NULL_DOUBLE ? "INVALID" : Utility::float_to_char(sqrt(rr)))<<"\n";
}

RegressionData::RegressionData(unsigned short _order)
{
	b  = NULL_DOUBLE;
	order = _order;
	xobs = NULL;
	yobs = NULL;
	ypred = NULL;
	n = 0;
	m = new double[order];
	for(int i=0; i<order; i++)
		m[i] = NULL_DOUBLE;
}

RegressionData::~RegressionData()
{
	delete ypred;
	delete m;
}

void RegressionData::set_m(unsigned short _order, double _m)
{
	if(_order <= order)
		m[_order-1] = _m;
	else
		Logging::log_debug("RegressionData::set_m(",Utility::int_to_char(_order),",",
							Utility::float_to_char(_m),") order out of range!",NULL);
}

double RegressionData::get_m(unsigned short _order) const
{
	if(_order <= order)
		return m[_order];
	else
	{
		Logging::log_debug("RegressionData::set_m(",Utility::int_to_char(_order),") order out of range!",NULL);
		return NULL_DOUBLE;
	}
}


void RegressionData::report(ostream& strm) const
{
	strm<<"y = "<<b<<" + ";
	if(order == 1)
		strm<<(m[0] != NULL_DOUBLE ? Utility::float_to_char(m[0]) : "INVALID")<<" x";
	else if(order == 2)
	{
		strm<<(m[0] != NULL_DOUBLE ? Utility::float_to_char(m[0]) : "INVALID")<<" x + ";
		strm<<(m[1] != NULL_DOUBLE ? Utility::float_to_char(m[1]) : "INVALID")<<" x^2";
	}
	else if(order == 3)
	{
		strm<<(m[0] != NULL_DOUBLE ? Utility::float_to_char(m[0]) : "INVALID")<<" x + ";
		strm<<(m[1] != NULL_DOUBLE ? Utility::float_to_char(m[1]) : "INVALID")<<" x^2 + ";
		strm<<(m[2] != NULL_DOUBLE ? Utility::float_to_char(m[2]) : "INVALID")<<" x^3";
	}
	else if(order == 4)
	{
		strm<<(m[0] != NULL_DOUBLE ? Utility::float_to_char(m[0]) : "INVALID")<<" x + ";
		strm<<(m[1] != NULL_DOUBLE ? Utility::float_to_char(m[1]) : "INVALID")<<" x^2 + ";
		strm<<(m[2] != NULL_DOUBLE ? Utility::float_to_char(m[2]) : "INVALID")<<" x^3 + ";
		strm<<(m[3] != NULL_DOUBLE ? Utility::float_to_char(m[3]) : "INVALID")<<" x^4";
	}
	else if(order == 5)
	{
		strm<<(m[0] != NULL_DOUBLE ? Utility::float_to_char(m[0]) : "INVALID")<<" x + ";
		strm<<(m[1] != NULL_DOUBLE ? Utility::float_to_char(m[1]) : "INVALID")<<" x^2 + ";
		strm<<(m[2] != NULL_DOUBLE ? Utility::float_to_char(m[2]) : "INVALID")<<" x^3 + ";
		strm<<(m[3] != NULL_DOUBLE ? Utility::float_to_char(m[3]) : "INVALID")<<" x^4 + ";
		strm<<(m[4] != NULL_DOUBLE ? Utility::float_to_char(m[4]) : "INVALID")<<" x^5";
	}
	else
		Logging::log_debug("RegressionData::report() order was greater ", Utility::int_to_char(MAX_ORDER), " cannot report!", NULL);
	strm<<"\n";
}

Statistics::Statistics(FocusRegistry* _registry)
{
	registry = _registry;
	registry->registerStatistics(this);
	init_critical_f_values();
}

RegressionData* Statistics::regression(double* _xobs, double* _yobs, int _n, unsigned short _order) throw (FocusException)
{
	Logging::log_debug("Statistics::regression(",Utility::int_to_char(_n),",",Utility::int_to_char(_order),")...",NULL);

	if(_order > MAX_ORDER)
		throw FocusException("Statistics::regression() order: ", Utility::int_to_char(_order)," but must be < 6. Aborted!");	
	if(_n < 2)
		throw FocusException("Statistics::regression() n: ", Utility::int_to_char(_order)," but must be > 2. Aborted!");
	
	RegressionData* reg_dat = new RegressionData(_order);
	reg_dat->set_n(_n);
	reg_dat->set_xobs(_xobs);
	reg_dat->set_yobs(_yobs);

	double* _ypred = new double[_n];
	double b, m1, cov00, cov01, cov11, sumsq;
	int i;

	if(_order == 1)
	{
		gsl_fit_linear (_xobs, 1, _yobs, 1, _n, &b, &m1, &cov00, &cov01, &cov11, &sumsq);

		for(i=0; i<_n; i++)
			_ypred[i] = b + (m1 * _xobs[i]);

		reg_dat->set_b(b);
		reg_dat->set_m(1,m1);
	}
	else if (_order > 0)
	{
		double chisq;
		gsl_matrix *X, *cov;
		gsl_vector *y, *c;

		X = gsl_matrix_alloc (_n, _order+1);
		cov = gsl_matrix_alloc (_order+1, _order+1);
		y = gsl_vector_alloc (_n);
		c = gsl_vector_alloc (_order+1);

#define C(i) (gsl_vector_get(c,(i)))

		for (i=0; i<_n; i++)
		{      
			for(int j=0; j<=_order; j++)
			{
				gsl_matrix_set(X, i, j, pow(_xobs[i],j));
			}
			gsl_vector_set (y, i, _yobs[i]);
		}

		gsl_multifit_linear_workspace* work = gsl_multifit_linear_alloc (_n, _order+1);
		gsl_multifit_linear (X, y, c, cov, &chisq, work);
		gsl_multifit_linear_free (work);

		reg_dat->set_b(C(0));

		for(i=1; i<=_order; i++)
			reg_dat->set_m(i,C(i));

		for(i=0; i<_n; i++)
		{
			double x = _xobs[i];
			if(_order == 2)
				_ypred[i] = C(0) + (C(1)*x) + (C(2)*pow(x,2));
			else if(_order == 3)
				_ypred[i] = C(0) + (C(1)*x) + (C(2)*pow(x,2)) + (C(3)*pow(x,3));
			else if(_order == 4)
				_ypred[i] = C(0) + (C(1)*x) + (C(2)*pow(x,2)) + (C(3)*pow(x,3)) + (C(4)*pow(x,4));
			else
				_ypred[i] = C(0) + (C(1)*x) + (C(2)*pow(x,2)) + (C(3)*pow(x,3)) + (C(4)*pow(x,4)) + (C(5)*pow(x,5));
		}
		
		gsl_matrix_free(X);
		gsl_matrix_free(cov);
		gsl_vector_free(c);
		gsl_vector_free(y);
	}
	reg_dat->set_ypred(_ypred);

	Logging::log_debug("Statistics::regression(",Utility::int_to_char(_n),",",Utility::int_to_char(_order),") DONE.",NULL);
	return reg_dat;
}

double Statistics::correlation(double* _x, double* _y, int _n)
{
	Logging::log_debug("Statistics::correlation(",Utility::int_to_char(_n),")...",NULL);

	double x_mean = gsl_stats_mean(_x,1,_n);
	double y_mean = gsl_stats_mean(_y,1,_n);

	double nominator = 0;
	double denominator_x = 0;
	double denominator_y = 0;
	double r = NULL_DOUBLE;
	double denominator, x_diff, y_diff;

	for(int i=0; i<_n; i++)
	{
		x_diff = _x[i] - x_mean;
		y_diff = _y[i] - y_mean;
		nominator += x_diff * y_diff;
		denominator_x += pow(x_diff,2);
		denominator_y += pow(y_diff,2);
	}
	
	denominator = sqrt(denominator_x * denominator_y);
	
	if(denominator != 0)
		r = nominator / sqrt(denominator_x * denominator_y);
	else
		Logging::log_debug("Statistics::correlation() denominator was 0!", NULL);

	Logging::log_debug("Statistics::correlation(",Utility::int_to_char(_n),") DONE.",NULL);	
	return r;
}

AnovaData* Statistics::anova(double* _yobs, double* _ypred, int _n, float _m_0)
{
	Logging::log_debug("Statistics::anova(",Utility::int_to_char(_n),")...",NULL);

	double y_mean = gsl_stats_mean(_yobs,1,_n);

	double y_pred, y_obs;
	double _sst = 0;
	double _ssr = 0;
	double _sse = 0;
	double _mse = 0;
	double _f = 0;
	double _rr = 0;
	double _r = 0;

	for(int i=0; i<_n; i++)
	{
		y_pred = _ypred[i];
		y_obs =  _yobs[i];

		_sst += pow(y_obs - y_mean, 2);
		_ssr += pow(y_pred - y_mean, 2);
		_sse += pow(y_obs - y_pred, 2);
	}

	if (_n > 2)
		_mse = _sse / (_n - 2);
	else
	{
		_mse = NULL_DOUBLE;
		Logging::log_debug("Statistics::anova(",Utility::int_to_char(_n),") denominator for MSE calculation was 0!", NULL);
	}

	if(_mse != 0 && _mse != NULL_DOUBLE)
		_f = _ssr / _mse;
	else
	{
		_f = NULL_DOUBLE;
		Logging::log_debug("Statistics::anova(",Utility::int_to_char(_n),") denominator MSE for F calculation was 0!", NULL);
	}

	if(_sst != 0)
	{
		_rr = _ssr / _sst;
		_r = sqrt(_rr);
	}
	else
	{
		_rr = NULL_DOUBLE;
		_r = NULL_DOUBLE;
		Logging::log_debug("Statistics::anova(",Utility::int_to_char(_n),") denominator SST for R-Square calculation was 0!", NULL);
	}

	if(_m_0 < 0 && _m_0 != NULL_DOUBLE)
		_r = _r * -1;

	AnovaData* anov_dat = new AnovaData();
	anov_dat->set_f(_f);
	anov_dat->set_mse(_mse);
	anov_dat->set_n(_n);
	anov_dat->set_r(_r);
	anov_dat->set_rr(_rr);
	anov_dat->set_sse(_sse);
	anov_dat->set_ssr(_ssr);
	anov_dat->set_sst(_sst);
	
	Logging::log_debug("Statistics::anova(",Utility::int_to_char(_n),") DONE.",NULL);
	return anov_dat;
}

unsigned short Statistics::functional_stability_test(double* _xobs, double* _yobs, int _n, unsigned short _order)
{
	Logging::log_debug("Statistics::functional_stability_test(",Utility::int_to_char(_n),",",Utility::int_to_char(_order),")...",NULL);

	if(_order == NULL_DOUBLE || _order > MAX_ORDER)
	{
		Logging::log_debug("Statistics::functional_stability_test() order out of range! Aborted.",NULL);
		return NULL_SHORT;
	}

	if(_n <= _order)
	{
		Logging::log_debug("Statistics::functional_stability_test() n < order! Aborted.",NULL);
		return NULL_SHORT;
	}

	unsigned short best_order = 1;

	for(unsigned short i=2; i<=_order; i++)
	{
		RegressionData* reg_dat = regression(_xobs, _yobs, _n, i);
		float m_0 = reg_dat->get_m(0);
		AnovaData* anov_dat = anova(_yobs,reg_dat->get_ypred(),_n, m_0);
		if(anov_dat->get_f() > get_critical_f_value(_n-2, 1))
			best_order++;
		delete reg_dat;
		delete anov_dat;
	}
	Logging::log_debug("Statistics::functional_stability_test(",Utility::int_to_char(_n),",",Utility::int_to_char(_order),") DONE.",NULL);
	return best_order;
}

void Statistics::init_critical_f_values()
{
	double critical_f_vals[34][19] = 
	{
		{161.4,199.5,215.7,224.6,230.2,234,236.8,238.9,240.5,241.9,243.9,245.9,248,249.1,250.1,251.1,252.2,253.3,254.3},
		{18.51,19,19.16,19.25,19.3,19.33,19.35,19.37,19.38,19.4,19.41,19.43,19.45,19.45,19.46,19.47,19.48,19.49,19.5},
		{10.13,9.55,9.28,9.12,9.01,8.94,8.89,8.85,8.81,8.79,8.74,8.7,8.66,8.64,8.62,8.59,8.57,8.55,8.53},
		{7.71,6.94,6.59,6.39,6.26,6.16,6.09,6.04,6,5.96,5.91,5.86,5.8,5.7,5.75,5.72,5.69,5.66,5.63},
		{6.61,5.79,5.41,5.19,5.05,4.95,4.88,4.82,4.77,4.74,4.68,4.62,4.56,4.53,4.5,4.46,4.43,4.4,4.36},
		{5.99,5.14,4.76,4.53,4.39,4.28,4.21,4.15,4.1,4.06,4,3.94,3.87,3.84,3.81,3.77,3.74,3.7,3.67},
		{5.59,4.74,4.35,4.12,3.97,3.87,3.79,3.73,3.68,3.64,3.57,3.51,3.44,3.41,3.38,3.34,3.3,3.27,3.23},
		{5.32,4.46,4.07,3.84,3.69,3.58,3.5,3.44,3.39,3.35,3.28,3.22,3.15,312,3.08,3.04,3.01,2.97,2.93},
		{5.12,4.26,3.86,3.63,3.48,3.37,3.29,3.23,3.18,3.14,3.07,3.01,2.94,4.9,2.86,2.83,2.79,2.75,2.71},
		{4.96,4.1,3.71,3.48,3.33,3.22,3.14,3.07,3.02,2.98,2.91,2.85,2.77,4.74,2.7,2.66,2.62,2.58,2.54},
		{4.84,3.98,3.59,3.36,3.2,3.09,3.01,2.95,2.9,2.85,2.79,2.72,2.65,2.61,2.57,2.53,2.49,2.45,2.4},
		{4.75,3.89,3.49,3.26,3.11,3,2.91,2.85,2.8,2.75,2.69,2.62,2.54,2.51,2.47,2.43,2.38,2.34,2.3},
		{4.67,3.81,3.41,3.18,3.03,2.92,2.83,2.77,2.71,2.67,2.6,2.53,2.46,2.42,2.38,2.34,2.3,3.25,2.21},
		{4.6,3.74,3.34,3.11,2.96,2.85,2.76,2.7,2.65,2.6,2.53,2.46,2.39,2.35,2.31,2.27,2.22,2.18,2.13},
		{4.54,3.68,3.29,3.06,2.9,2.79,2.71,2.64,2.59,2.54,2.48,2.4,2.33,2.29,2.25,2.2,2.16,2.11,2.07},
		{4.49,3.63,3.24,3.01,2.85,2.74,2.66,2.59,2.54,2.49,2.42,2.35,2.28,2.24,2.19,2.15,2.11,2.06,2.01},
		{4.45,3.59,3.2,2.96,2.81,2.7,2.61,2.55,2.49,2.45,2.38,2.31,2.23,2.19,2.15,2.1,2.06,2.01,1.96},
		{4.41,3.55,3.16,2.93,2.77,2.66,2.58,2.51,2.46,2.41,2.34,2.27,2.19,2.15,2.11,2.06,2.02,1.97,1.92},
		{4.38,3.52,3.13,2.9,2.74,2.63,2.54,2.48,2.42,2.38,2.31,2.23,2.16,2.11,2.07,2.03,1.98,1.93,1.88},
		{4.35,3.49,3.1,2.87,2.71,2.6,2.51,2.45,2.39,2.35,2.28,2.2,2.12,2.08,2.04,1.99,1.95,1.9,1.84},
		{4.32,3.47,3.07,2.84,2.68,2.57,2.49,2.42,2.37,2.32,2.25,2.18,2.1,2.05,2.01,1.96,1.92,1.87,1.81},
		{4.3,3.44,3.05,2.82,2.66,2.55,2.46,2.4,2.34,2.3,2.23,2.15,2.07,2.03,1.98,1.94,1.89,1.84,1.78},
		{4.28,3.42,3.03,2.8,2.64,2.53,2.44,2.37,2.32,2.27,2.2,2.13,2.05,2.01,1.96,1.91,1.86,1.81,1.76},
		{4.26,3.4,3.01,2.78,2.62,2.51,2.42,2.36,2.3,2.25,2.18,2.11,2.03,1.98,1.94,1.89,1.84,1.79,1.73},
		{4.24,3.39,2.99,2.76,2.6,2.49,2.4,2.34,2.28,2.24,2.16,2.09,2.01,1.96,1.92,1.87,1.82,1.77,1.71},
		{4.23,3.37,2.98,2.74,2.59,2.47,2.39,2.32,227,2.22,2.15,2.07,1.99,1.95,1.9,1.85,1.8,1.75,1.69},
		{4.21,3.35,2.96,2.73,2.57,2.46,2.37,2.31,2.25,2.2,2.13,2.06,1.97,1.93,1.88,1.84,1.79,1.73,1.67},
		{4.2,3.34,2.95,2.71,2.56,2.45,2.36,2.29,2.24,2.19,2.12,2.04,1.96,1.91,1.87,1.82,1.77,1.71,1.65},
		{4.18,3.33,2.93,2.7,2.55,2.43,2.35,2.28,2.22,2.18,2.1,2.03,1.94,1.9,1.85,1.81,1.75,1.7,1.64},
		{4.17,3.32,2.92,2.69,2.53,2.42,2.33,2.27,2.21,2.16,2.09,2.01,1.93,1.89,1.84,1.79,1.74,1.68,1.62},
		{4.08,3.23,2.84,2.61,2.45,2.34,2.25,2.18,2.12,2.08,2,1.92,1.84,1.79,1.74,1.69,1.64,1.58,1.51},
		{4,3.15,2.76,2.53,2.37,2.25,2.17,2.1,2.04,1.99,1.92,1.84,1.75,1.7,1.65,1.59,1.53,1.47,1.39},
		{3.92,3.07,2.68,2.45,2.29,2.17,2.09,2.02,1.96,1.91,1.83,1.75,1.66,1.61,1.55,1.5,1.43,1.35,1.25},
		{3.84,3,2.6,2.37,2.21,2.1,2.01,1.94,1.88,1.83,1.75,1.67,1.57,1.52,1.46,1.39,1.32,1.22,1},
	};

	for(int dfdenom=0; dfdenom<34; dfdenom++)
		for(int dfnumer=0; dfnumer<19; dfnumer++)
			critical_f_values[dfdenom][dfnumer] = critical_f_vals[dfdenom][dfnumer];
}

double Statistics::get_critical_f_value(int _dfdenom, int _dfnumer) const
{
	if(_dfdenom > 0 && _dfdenom < 31)
	{
		if(_dfnumer > 0 && _dfnumer < 11)
			return critical_f_values[_dfdenom-1][_dfnumer-1];
		if(_dfnumer > 10 && _dfnumer < 13)
			return critical_f_values[_dfdenom-1][10];
		if(_dfnumer > 12 && _dfnumer < 16)
			return critical_f_values[_dfdenom-1][11];
		if(_dfnumer > 15 && _dfnumer < 21)
			return critical_f_values[_dfdenom-1][12];
		if(_dfnumer > 20 && _dfnumer < 25)
			return critical_f_values[_dfdenom-1][13];
		if(_dfnumer > 24 && _dfnumer < 31)
			return critical_f_values[_dfdenom-1][14];
		if(_dfnumer > 30 && _dfnumer < 41)
			return critical_f_values[_dfdenom-1][15];
		if(_dfnumer > 40 && _dfnumer < 61)
			return critical_f_values[_dfdenom-1][16];
		if(_dfnumer > 60 && _dfnumer < 121)
			return critical_f_values[_dfdenom-1][17];
		if(_dfnumer > 120)
			return critical_f_values[_dfdenom-1][18];
	}
	else if(_dfdenom > 30 && _dfdenom < 41)
	{
		if(_dfnumer > 0 && _dfnumer < 11)
			return critical_f_values[30][_dfnumer-1];
		if(_dfnumer > 10 && _dfnumer < 13)
			return critical_f_values[30][10];
		if(_dfnumer > 12 && _dfnumer < 16)
			return critical_f_values[30][11];
		if(_dfnumer > 15 && _dfnumer < 21)
			return critical_f_values[30][12];
		if(_dfnumer > 20 && _dfnumer < 25)
			return critical_f_values[30][13];
		if(_dfnumer > 24 && _dfnumer < 31)
			return critical_f_values[30][14];
		if(_dfnumer > 30 && _dfnumer < 41)
			return critical_f_values[30][15];
		if(_dfnumer > 40 && _dfnumer < 61)
			return critical_f_values[30][16];
		if(_dfnumer > 60 && _dfnumer < 121)
			return critical_f_values[30][17];
		if(_dfnumer > 120)
			return critical_f_values[30][18];
	}
	else if(_dfdenom > 40 && _dfdenom < 61)
	{
		if(_dfnumer > 0 && _dfnumer < 11)
			return critical_f_values[31][_dfnumer-1];
		if(_dfnumer > 10 && _dfnumer < 13)
			return critical_f_values[31][10];
		if(_dfnumer > 12 && _dfnumer < 16)
			return critical_f_values[31][11];
		if(_dfnumer > 15 && _dfnumer < 21)
			return critical_f_values[31][12];
		if(_dfnumer > 20 && _dfnumer < 25)
			return critical_f_values[31][13];
		if(_dfnumer > 24 && _dfnumer < 31)
			return critical_f_values[31][14];
		if(_dfnumer > 30 && _dfnumer < 41)
			return critical_f_values[31][15];
		if(_dfnumer > 40 && _dfnumer < 61)
			return critical_f_values[31][16];
		if(_dfnumer > 60 && _dfnumer < 121)
			return critical_f_values[31][17];
		if(_dfnumer > 120)
			return critical_f_values[31][18];
	}
	else if(_dfdenom > 60 && _dfdenom < 121)
	{
		if(_dfnumer > 0 && _dfnumer < 11)
			return critical_f_values[32][_dfnumer-1];
		if(_dfnumer > 10 && _dfnumer < 13)
			return critical_f_values[32][10];
		if(_dfnumer > 12 && _dfnumer < 16)
			return critical_f_values[32][11];
		if(_dfnumer > 15 && _dfnumer < 21)
			return critical_f_values[32][12];
		if(_dfnumer > 20 && _dfnumer < 25)
			return critical_f_values[32][13];
		if(_dfnumer > 24 && _dfnumer < 31)
			return critical_f_values[32][14];
		if(_dfnumer > 30 && _dfnumer < 41)
			return critical_f_values[32][15];
		if(_dfnumer > 40 && _dfnumer < 61)
			return critical_f_values[32][16];
		if(_dfnumer > 60 && _dfnumer < 121)
			return critical_f_values[32][17];
		if(_dfnumer > 120)
			return critical_f_values[32][18];
	}
	else if(_dfdenom > 120)
	{
		if(_dfnumer > 0 && _dfnumer < 11)
			return critical_f_values[33][_dfnumer-1];
		if(_dfnumer > 10 && _dfnumer < 13)
			return critical_f_values[33][10];
		if(_dfnumer > 12 && _dfnumer < 16)
			return critical_f_values[33][11];
		if(_dfnumer > 15 && _dfnumer < 21)
			return critical_f_values[33][12];
		if(_dfnumer > 20 && _dfnumer < 25)
			return critical_f_values[33][13];
		if(_dfnumer > 24 && _dfnumer < 31)
			return critical_f_values[33][14];
		if(_dfnumer > 30 && _dfnumer < 41)
			return critical_f_values[33][15];
		if(_dfnumer > 40 && _dfnumer < 61)
			return critical_f_values[33][16];
		if(_dfnumer > 60 && _dfnumer < 121)
			return critical_f_values[33][17];
		if(_dfnumer > 120)
			return critical_f_values[33][18];
	}
	else
	{
		Logging::log_debug("Statistics::get_critical_f_value(", Utility::int_to_char(_dfdenom),",",
																Utility::int_to_char(_dfnumer), ") could not retrieve value!",NULL);
	}
	return NULL_DOUBLE;
}




#endif