/*FOCUS-2.0 is a sampling and statistical data analysis program
for spatial data sets. Copyright (C) 2003  Lutz Tischendorf

This program is free software; you can redistribute it
and/or modify it under the terms of the GNU General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public
License along with this program; if not, write to the Free
Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.*/

#ifndef __SAMPLER_CPP
#define __SAMPLER_CPP

#include "sampler.h"
#include "logging.h"
#include "focusregistry.h"
#include "focusutil.h"
#include "plot.h"
#include "distance.h"
#include <algorithm>


Sampler::Sampler(FocusRegistry* _registry)
{
	registry = _registry;
	registry->registerSampler(this);
	allowDuplicateSamples = false;
	enforceUniqueSamples = false;
	maxPlots = USHRT_MAX;
	maxDuplicateTrials = 1000;
}

Sampler::Sampler(FocusRegistry* _registry, unsigned short _maxPlots)
{
	registry = _registry;
	registry->registerSampler(this);
	allowDuplicateSamples = false;
	enforceUniqueSamples = false;
	maxPlots = _maxPlots;
	maxDuplicateTrials = 1000;
}

Sampler::Sampler(FocusRegistry* _registry, unsigned short _maxPlots, bool _dupl, bool _unique)
{
	registry = _registry;
	registry->registerSampler(this);
	allowDuplicateSamples = _dupl;
	enforceUniqueSamples = _unique;
	maxPlots = _maxPlots;
	maxDuplicateTrials = 1000;
}

Sampler::Sampler(FocusRegistry* _registry, unsigned short _maxPlots, unsigned int _maxDuplicateTrials, bool _dupl, bool _unique)
{
	registry = _registry;
	registry->registerSampler(this);
	allowDuplicateSamples = _dupl;
	enforceUniqueSamples = _unique;
	maxPlots = _maxPlots;
	maxDuplicateTrials = _maxDuplicateTrials;
}

Sampler::~Sampler()
{
	sampleHistory.clear();
}

void Sampler::sampleByDistance(double _distance, sample_types _sample_type, 
							   variable_types _variable_type, plotset_types _plotset_type) throw (FocusException)
{
	Logging::log_debug("Sampler::sampleByDistance(", Utility::float_to_char(_distance), ",", 
		(_sample_type == INSIDE ? "inside" : "outside"), ") ...",NULL);

	const char* ps_type;

	if(_plotset_type == PLOT_BUFFER)
		ps_type = PS_PLOT_BUFFER;
	else
		ps_type = (_variable_type == PREDICTOR ? PS_PLOT_PLOT_PREDICTOR : PS_PLOT_PLOT_RESPONSE);

	PlotList* plots = registry->getPlotSetList()->getPlotSetForReferenceName(ps_type)->getPlotList();
	DistanceList* distances = registry->getPlotSetList()->getPlotSetForReferenceName(ps_type)->getDistanceList();

	sampleHistory.push_back(*(new plot_list()));
	sampleIterator = sampleHistory.end();
	sampleIterator--;
	unsigned int numberOfDuplicateTrials = 0;

START:
	numberOfDuplicateTrials++;
	unsigned short sampledPlots = 0;
	plots->selectAllPlots();
	random_shuffle(plots->begin(), plots->end());
	PlotList::iterator plot_iterator = plots->begin();
	Plot* currentPlot;
	
	while(plot_iterator != plots->end())
	{
		currentPlot = *plot_iterator;
		if((currentPlot)->isSelected())
		{
			if(sampledPlots < maxPlots)
			{
				if(_sample_type == OUTSIDE)
					distances->unselectPlotsWithinDistance(currentPlot,_distance);
				else
					distances->unselectPlotsOutsideDistance(currentPlot,_distance);

				(*sampleIterator).push_back((char*)currentPlot->getId());
				sampledPlots++;
			}
			else
				currentPlot->unSelect();
		}
		plot_iterator++;

		//no overlap, each sample is a unique set of plots
		if(enforceUniqueSamples)
		{
			if(checkForIdenticalSample(sampleIterator))
			{
				(*sampleIterator).pop_back();
				currentPlot->unSelect();
				sampledPlots--;
			}
			if(plot_iterator == plots->end() && sampledPlots < maxPlots-1)
			{
				Logging::log_error("Sampling aborted after ", Utility::int_to_char(sampleHistory.size()-1), " iterations.",NULL);
				throw FocusException("Sampler::sampleByDistance() could not find unique sample for distance '",Utility::float_to_char(_distance), "'");
			}
		}

		//no duplicates, still allows same plots in different samples
		if(plot_iterator == plots->end() && !allowDuplicateSamples && !enforceUniqueSamples)
		{
			if(numberOfDuplicateTrials <= maxDuplicateTrials)
			{
				sort((*sampleIterator).begin(), (*sampleIterator).end(), Greater_Char());
				if(checkForDuplicateSample(sampleIterator))
				{
					(*sampleIterator).clear();
					goto START;
				}
			}
			else
			{
				Logging::log_error("Sampling aborted after ", Utility::int_to_char(sampleHistory.size()), " iterations.",NULL);
				throw FocusException("Sampler::sampleByDistance() could not find unique sample for distance '",
					Utility::float_to_char(_distance), "' after number of trials ", Utility::int_to_char(maxDuplicateTrials));
			}
		}
	}
	Logging::log_debug("Sampler::sampleByDistance() DONE.",NULL);	
}

bool Sampler::checkForIdenticalSample(sample_history_iterator& _sampleIterator)
{	
	sample_history_list::iterator histIterator = sampleHistory.begin();
	plot_list::iterator samplePlotIterator = _sampleIterator->begin();
	plot_list::iterator histPlotIterator = (*histIterator).begin();

	while(histIterator != sampleHistory.end())
	{
		if(histIterator != _sampleIterator)
		{
			histPlotIterator = (*histIterator).begin();
			while(histPlotIterator != (*histIterator).end())
			{
				samplePlotIterator = _sampleIterator->begin();
				while(samplePlotIterator != _sampleIterator->end())
				{
					if(!strcmp(*samplePlotIterator++, *histPlotIterator))
						return true;
				}
				histPlotIterator++;
			}
		}
		histIterator++;
	}
	return false;
}

bool Sampler::checkForDuplicateSample(sample_history_iterator& _sampleIterator)
{	
	sample_history_list::iterator histIterator = sampleHistory.begin();
	while(histIterator != sampleHistory.end())
	{
		if(histIterator != _sampleIterator)
		{
			if((*histIterator) == (*sampleIterator))
				return true;
		}
		histIterator++;
	}
	return false;
}

void Sampler::reportSampleHistory(ostream& strm) const
{
	sample_history_list::const_iterator histIterator = sampleHistory.begin();
	plot_list::const_iterator plotIterator = (*histIterator).begin();
	int i = 1;
	while(histIterator != sampleHistory.end())
	{
		strm<<i++<<". [";
		plotIterator = (*histIterator).begin();	
		while(plotIterator != (*histIterator).end())
		{
			strm<<(*plotIterator++);
			if(plotIterator != (*histIterator).end())
				strm<<",";
		}
		strm<<"]\n";
		histIterator++;
	}
}

void Sampler::resetSampleHistory()
{
	sample_history_list::iterator histIterator = sampleHistory.begin();
	while(histIterator != sampleHistory.end())
	{
		(*histIterator).clear();
		histIterator++;
	}
	sampleHistory.clear();
}

#endif