/*FOCUS-2.0 is a sampling and statistical data analysis program
for spatial data sets. Copyright (C) 2003  Lutz Tischendorf

This program is free software; you can redistribute it
and/or modify it under the terms of the GNU General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public
License along with this program; if not, write to the Free
Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.*/

#ifndef __PLOT_CPP
#define __PLOT_CPP

#include "focusregistry.h"
#include "plot.h"

Plot::Plot(const char* _id)
{
	char* id_ = new char(100);
	sprintf(id_,"%s",_id);
	id = id_;
	x = NULL_DOUBLE;
	y = NULL_DOUBLE;
	selected = false;
}

Plot::Plot(const char* _id, double _x, double _y)
{
	char* id_ = new char(100);
	sprintf(id_,"%s",_id);
	id = id_;
	x = _x;
	y = _y;
	selected = false;
}

Plot::~Plot()
{
}

bool Plot::operator==(const Plot& _plot) const
{
	return !strcmp(id, _plot.id);
}

bool Plot::operator!=(const Plot& _plot) const
{
	return !((*this)==_plot);
}

void Plot::addResponse_variable(Variable& _variable)
{
	response_variables.push_back(_variable);
}

void Plot::addPredictor_variable(Variable& _variable)
{
	predictor_variables.push_back(_variable);
}

Variable Plot::getResponseVariableAtIndex(int _variable_index) const
{
	return response_variables[_variable_index];
}

Variable Plot::getPredictorVariableAtIndex(int _variable_index) const
{
	return predictor_variables[_variable_index];
}

unsigned short Plot::getPredictorVariableIndexByContext(double _distance, sample_types _sample_type) const
{
	int number_of_variables = predictor_variables.size();

	double neg_diff = 1000000;
	int	neg_index = 0;
	double pos_diff = 1000000;
	int pos_index = 0;

	for(int i=0; i<number_of_variables; i++)
	{
		double tmpValue = ((CircularPlotBuffer*)predictor_variables[i].getSpatialContext())->getDistance();
		if((_sample_type == OUTSIDE) && (tmpValue <= _distance) && ((tmpValue - _distance) <= neg_diff))
		{
			neg_diff = _distance - tmpValue;
			neg_index = i;
		}
		else if((_sample_type == INSIDE) && (tmpValue >= _distance) && ((tmpValue - _distance) <= pos_diff))
		{
			pos_diff = tmpValue - _distance;
			pos_index = i;
		}
	}
	return _sample_type == OUTSIDE ? neg_index : pos_index;
}

void Plot::report(ostream& strm) const
{
	strm<<"Plot(id="<<id<<") x="<<(x == NULL_DOUBLE ? 0 : x); 
	strm<<",y="<<(y == NULL_DOUBLE ? 0 : y);
	strm<<",selected="<<(selected == false ? "no" : "yes");
	strm<<")\n";
	variable_list::const_iterator predvar_iterator = predictor_variables.begin();
	while(predvar_iterator != predictor_variables.end())
		(*predvar_iterator++).report(strm);
	variable_list::const_iterator respvar_iterator = response_variables.begin();
	while(respvar_iterator != response_variables.end())
		(*respvar_iterator++).report(strm);

}

PlotList::PlotList()
{
}

PlotList::~PlotList()
{
	PlotList::iterator plot_iterator = begin();
	while(plot_iterator != end())
	{
		Plot* tmpPlot = *plot_iterator++;
		delete tmpPlot;
	}
	clear();
}

void PlotList::selectAllPlots()
{
	PlotList::iterator plot_iterator = begin();
	while(plot_iterator != end())
		(*plot_iterator++)->select();
}

void PlotList::addPlot(Plot* _plot)
{
	push_back(_plot);
}

Plot* PlotList::findPlot(char* _id) const
{
	PlotList::const_iterator plot_iterator = begin();
	while(plot_iterator != end())
	{
		if(!strcmp((*plot_iterator)->getId(),_id))
			return *plot_iterator;
		plot_iterator++;
	}
	return NULL;
}

int PlotList::getNumberOfSelectedPlots() const
{
	int numberOfSelectedPlots = 0;

	PlotList::const_iterator plot_iterator = begin();
	while(plot_iterator != end())
	{
		if((*plot_iterator++)->isSelected())
			numberOfSelectedPlots++;
	}
	return numberOfSelectedPlots;
}

double* PlotList::getResponseVariableValuesForSelectedPlots(unsigned short _variable_index) const
{
	double* values = new double[getNumberOfSelectedPlots()];
	int index = 0;

	PlotList::const_iterator plot_iterator = begin();
	while(plot_iterator != end())
	{
		if((*plot_iterator)->isSelected())
			values[index++] = (*plot_iterator)->getResponseVariableAtIndex(_variable_index).getValue();
		plot_iterator++;
	}
	return values;
}

double* PlotList::getPredictorVariableValuesForSelectedPlots(unsigned short _variable_index) const
{
	double* values = new double[getNumberOfSelectedPlots()];
	int index = 0;

	PlotList::const_iterator plot_iterator = begin();
	while(plot_iterator != end())
	{
		if((*plot_iterator)->isSelected())
			values[index++] = (*plot_iterator)->getPredictorVariableAtIndex(_variable_index).getValue();
		plot_iterator++;
	}
	return values;
}

int PlotList::getPredictorVariableIndexByContext(double _distance, sample_types _sample_type) const
{
	PlotList::const_iterator plot_iterator = begin();
	if(plot_iterator != end())
			return (*plot_iterator)->getPredictorVariableIndexByContext(_distance, _sample_type);
	return NULL_SHORT;
}		

void PlotList::report(ostream& strm) const
{
	PlotList::const_iterator plot_iterator = begin();
	while(plot_iterator != end())
		(*plot_iterator++)->report(strm);
}

void PlotList::reportSelectedPlotIds(ostream& strm) const
{
	PlotList::const_iterator plot_iterator = begin();
	while(plot_iterator != end())
	{
		if((*plot_iterator)->isSelected())
		{
			strm<<(*plot_iterator)->getId();
			strm<<",";
		}
		plot_iterator++;
	}
}

int PlotList::reportNumberofSelectedPlots(ostream& strm) const
{
	int counter = 0;
	PlotList::const_iterator plot_iterator = begin();
	while(plot_iterator != end())
	{
		if((*plot_iterator++)->isSelected())
			counter++;
	}
	strm<<counter<<" Plots are selected.\n";
	return counter;
}

PlotSet::PlotSet(PlotList* _plots, DistanceList* _distances, categories _category, const char* _name)
{
	plots = _plots;
	distances = _distances;
	category = _category;
	reference_name = _name;
	constraints = NULL;
}
	
PlotSet::~PlotSet()
{
}

PlotSetList::PlotSetList(FocusRegistry* _registry)
{
	registry = _registry;
	registry->registerPlotSetList(this);
}

PlotSetList::~PlotSetList()
{
}

void PlotSetList::addPlotSet(PlotSet* _plotSet)
{
	push_back(_plotSet);
}

PlotSet* PlotSetList::getPlotSetForReferenceName(const char* _name) const
{
	PlotSetList::const_iterator iterator = begin();
	while(iterator != end())
	{
		if(!strcmp((*iterator)->getReferenceName(), _name))
			return *iterator;
		iterator++;
	}
	return NULL;
}

#endif