/*FOCUS-2.0 is a sampling and statistical data analysis program
for spatial data sets. Copyright (C) 2003  Lutz Tischendorf

This program is free software; you can redistribute it
and/or modify it under the terms of the GNU General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public
License along with this program; if not, write to the Free
Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.*/

#ifndef __DYNFLOATARRAY_CPP
#define __DYNFLOATARRAY_CPP

#include "defs.h"
#include "dfarr.h"
#include <stdlib.h>
#include <math.h>

//******************************************************************************

dfloatlist::dfloatlist()
{
        leftfloat=NULL;
        rightfloat=NULL;
}

dfloatlist::dfloatlist(int x)
{
	leftfloat=NULL;
	rightfloat=NULL;
	for(int i=0; i<x; i++)
		add(NULL_FLOAT);
}

dfloatlist::dfloatlist(dfloatlist& flist)
{
	leftfloat=NULL;
	rightfloat=NULL;
	int listsize=flist.entries();
        for(int i=0; i<listsize; i++)
                add(flist[i]);
}

dfloatlist::~dfloatlist()
{
	reset();
}

void dfloatlist::reset()
{
    float_el* current=leftfloat;
    float_el* next;
	while(current)
	{
        next=current->right;
        delete current;
        current=next;
    }
    leftfloat=NULL;
	rightfloat=NULL;
}

void dfloatlist::add(float f)
{
	float_el* new_el=new float_el;
	if(!new_el)
	{
		error(1);
		exit(1);
	}
	if(!leftfloat)
		leftfloat=new_el;
	new_el->fval=f;
	if(rightfloat)
		rightfloat->right=new_el;
	new_el->right=NULL;
	rightfloat=new_el;
}

float& dfloatlist::operator[](int x)
{
	float_el* current=leftfloat;
    int count=0;
    while(current)
    {
        if(count==x)
			return current->fval;
        current=current->right;
        count++;
    }
    error(3,(float)x);
	float* retval=new float(-1); 
        return *retval;
}

int dfloatlist::entries() const
{
    float_el* current=leftfloat;
	int count=0;
    while(current)
    {
		if(current->fval != NULL_FLOAT)
			count++;
        current=current->right;
    }
    return count;
}

float dfloatlist::last_el() const
{
    float_el* current=leftfloat;
    float val=0;
    while(current)
    {
        val=current->fval;
        current=current->right;
    }
    return val;
}

int dfloatlist::index_of(float val) const
{
    float_el* current=leftfloat;
    int count=0;
    while(current)
    {
        if(current->fval==val)
			return count;
        count++;
        current=current->right;
    }
    return -1;
}

int dfloatlist::operator=(dfloatlist& flist) 
{
	int listsize=flist.entries();
	reset();
	for(int i=0; i<listsize; i++)
		add((flist[i]));
	return 1;
}

int dfloatlist::operator==(dfloatlist& flist)
{
	int listsize=flist.entries();
    if(listsize!=entries())
	return 0;
    for(int i=0; i<listsize; i++)
	{
		if((*this)[i]!=flist[i])
			return 0;
	}
	return 1;
}

int dfloatlist::operator!=(dfloatlist& flist)
{
	if(*this==flist)
		return 0;
	return 1;
}

void dfloatlist::report(ostream& strm) const
{
	float_el* current=leftfloat;
	while(current)
	{
		if(current->fval == NULL_FLOAT)
			strm<<"INVALID"<<" ";
		else
			strm<<current->fval<<" ";
		current=current->right;
	}
}

void dfloatlist::error(int n, float val) const
{
	cout<<"ERROR CLASS dfloatlist ::";
	switch(n)
   {
		case 1: 	cout<<"add(): lack of memory!"<<endl; break;
		case 2: 	cout<<"set(): position: "<< (int)val<< " out of permitted range: ";
					cout<<entries()<<" !"<<endl;
					break;
		case 3: 	cout<<"[]: index: "<< (int)val<< " out of range: ";
   					cout<<"[0.."<<entries()<<"]!"<<endl;
					break;
		default: cout << "error(): no error et all?" << endl;
	}
}

float dfloatlist::mean() const
{
	float_el* current=leftfloat;
	float sum=0;
	int count=0;
    while(current)
    {
		if(current->fval != NULL_FLOAT)
		{
			sum+=current->fval;
			count++;
		}
		current=current->right;
    }
	if(count>0)
		return sum/=float(count);
	return 0;
}

float dfloatlist::min() const
{
    float_el* current=leftfloat; 
    float min=65000;
    while(current) 
    { 
		if(current->fval != NULL_FLOAT && current->fval < min)
			min=current->fval;
        current=current->right;
    }
	if(min != 65000) 
		return min;
	return 0;
}

float dfloatlist::max() const 
{ 
    float_el* current=leftfloat;  
    float max=-65000;
    while(current) 
    {  
		if(current->fval != NULL_FLOAT && current->fval > max)
			max=current->fval;
		current=current->right;  
    }  
    return max; 
}

float dfloatlist::varianz() const
{
	float_el* current=leftfloat;
	double sum=0;
	int count=0;
	float xq=mean();
	while(current)
	{
		if(current->fval != NULL_FLOAT)
		{
			sum+=pow((current->fval-xq),2);
			count++;
		}
		current=current->right;
	}
	if(count>1)
		return (float)(sum/=(double)(count));
	return 0;
}

float dfloatlist::stderro() const
{
	int count=entries();
	if(count>0)
		return (float)((sqrt(varianz())/sqrt((float)count)));
	return 0;
}

float dfloatlist::confidence_interval() const
{
	int count=entries();
	if(count>0)
		return (float)(1.96*(sqrt(varianz())/sqrt((float)count)));
	return 0;
}

//sorts elements of the list in descending order

void dfloatlist::desc_sort()
{
	int size=entries();
	int sortstep=0;
	int index, minindex; 
	float min;
	while(sortstep<size)
	{
		min=65000;
		minindex=0;
		float_el* current=leftfloat;
		//find the largest value which has not yet been sorted
		for(index=0; index<size; index++)
		{
			if(index>=sortstep && current->fval<min)
			{
				min=current->fval;
				minindex=index;
			}
			current=current->right;
		}
		index=0;
		current=leftfloat;
		float_el* prev_el=NULL;
		//second iteration to relocate (switching pointers) the max element
		while(index<minindex)
		{
			prev_el=current; 
                        current=current->right;  
                        index++;
		}
		if(prev_el)
		{
			prev_el->right=current->right;
			current->right=leftfloat;
			leftfloat=current;
		}
		sortstep++;
	}
}

//sorts elements of the list in ascending order

void dfloatlist::asc_sort()
{
    int size=entries();
    int sortstep=0;
    int index, maxindex;
    float max;
    while(sortstep<size)
    {
        max=0;
        maxindex=0;
        float_el* current=leftfloat;
        //find the largest value which has not yet been sorted
        for(index=0; index<size; index++)
        {
            if(index>=sortstep && current->fval>=max)
            {
                    max=current->fval;
                    maxindex=index;
            }
            current=current->right;
        }
        index=0; 
        current=leftfloat;
        float_el* prev_el=NULL;
        //second iteration to relocate (switching pointers) the max element
        while(index<maxindex) 
        {
            prev_el=current; 
            current=current->right;
            index++;
        }
        if(prev_el)
        { 
            prev_el->right=current->right;
            current->right=leftfloat;
            leftfloat=current;
        }
        sortstep++;
    }
}

//******************************************************************************

floatlist::floatlist(int x)
{
    size=x;
    flist=new float[size];
    if(!flist)
    {
        error(1);
        exit (1);
    }
    for(int i=0; i<size; i++)
		flist[i]=NULL_FLOAT;
}

floatlist::floatlist(floatlist& fl)
{
    size=fl.get_size();
    flist=new float[size];
    if(!flist)
    {
        error(1);
        exit (1);
    }
    for (int i=0; i<size; i++)
		flist[i]=fl[i];
}

floatlist::~floatlist()
{
	reset();
}

void floatlist::reset()
{
	delete flist;
}

int floatlist::entries() const
{
	int count = 0;
	for(int i=0; i<size; i++)
	{
		if(flist[i] != NULL_FLOAT)
			count++;
	}
	return count;
}

float& floatlist::operator[](int x)
{
    if(x<size)
		return flist[x];
    error(2,(float)x);
    float* dummy=new float(-1);
    return *dummy;
}
 
int floatlist::operator=(floatlist& fl)
{
    delete flist;
    size=fl.get_size();
    flist=new float[size];
    if(!flist)
    {
        error(3);
        exit (1);
    }
    for (int i=0; i<size; i++)
		flist[i]=fl[i];
    return 1;
}

int floatlist::index_of(float val) const
{
    for(int i=0; i<size; i++)
		if(flist[i]==val)
			return i;
    return -1;
}
 
int floatlist::operator==(floatlist& fl) const
{
    if(size!=fl.get_size())
		return 0;
    for(int i=0; i<size; i++)
    {
		if(flist[i]!=fl[i])
			return 0;
    }
    return 1;
}

int floatlist::operator!=(floatlist& fl) const
{
    if(*this==fl)
		return 0;
    return 1;
}
 
void floatlist::report(ostream& strm) const
{
	for(int i=0; i<size; i++)
	{
		if(flist)
		{
			if(flist[i] == NULL_FLOAT)
				strm<<"INVALID,";
			else
				strm<<flist[i]<<",";
		}
   }
}
 
void floatlist::error(int n, float val) const
{
    cout<<"CLASS floatlist ::";
    switch(n) 
	{
		case 1: cout<<"floatlist(): lack of memory!" << endl;
				break;
		case 2: cout<<"[]: index: "<< (int)val<< " out of range: ";
				cout<<"[0.."<<size<<"]!"<<endl;
				getchar();
				break;
		case 3: cout<<"operator=(): lack of memory!" << endl;
				break;
		default: cout << "error(): no error et all?" << endl;
    }
}

float floatlist::mean() const
{
	if(flist && size>0)
	{
		float sum=0;
		int count=0;
		for(int i=0; i<size; i++)
		{
			if(flist[i] != NULL_FLOAT)
			{
				sum+=flist[i];
				count++;
			}
		}
		if(count>0)
			return sum/=float(count);
		return 0;
	}
	return 0;
}

float floatlist::min() const
{
    if(flist && size>0) 
    {
		float min=65000; 
		for(int i=0; i<size; i++) 
		{
			if(flist[i] != NULL_FLOAT && flist[i] < min)
				min=flist[i];
		}
		return min;
    } 
	return 0;
}

float floatlist::max() const
{
	if(flist && size>0)  
    { 
        float max=-65000;  
        for(int i=0; i<size; i++) 
		{
            if(flist[i] != NULL_FLOAT && flist[i] > max) 
				max=flist[i];
		}
        return max;
    } 
	return 0;
}

float floatlist::varianz() const
{
	if(flist && size>0)   
    { 
		double sum=0;
		int count=0;
		float xq=mean();
		for(int i=0; i<size; i++)
		{
			if(flist[i] != NULL_FLOAT)
			{
				sum+=pow((flist[i]-xq),2);
				count++;
			}
		}
		if(count>1)
			return (float)(sum/=(float)(count));
		return 0;
	}
	return 0;
}

float floatlist::stderro() const
{
	int count = entries();
	if(flist && count>0)
		return (float)((sqrt(varianz())/sqrt((float)count)));
	return 0;
}

float floatlist::confidence_interval() const
{
	int count = entries();
	if(flist && count>0)
		return (float)(1.96*(sqrt(varianz())/sqrt((float)count)));
	return 0;
}
		
//******************************************************************************

dfloatarray::dfloatarray()
{
	toplist=NULL;
	bottomlist=NULL;
}

dfloatarray::dfloatarray(int x, int y)
{
	toplist=NULL;
        bottomlist=NULL;
	for(int i=0; i<y; i++)
		add(new floatlist(x));
}

dfloatarray::dfloatarray(dfloatarray& farr)
{	
        toplist=NULL;
        bottomlist=NULL;
        int listsize=farr.entries();
        for(int i=0; i<listsize; i++)
                add(new floatlist(farr[i]));
}

dfloatarray::~dfloatarray()
{
	reset();
}

void dfloatarray::reset()
{
	floatlist_el* current=toplist;
	floatlist_el* next;
	while(current)
	{
		next=current->right;
		delete current->flval;
		delete current;
		current=next;
	}
	toplist=NULL;
	bottomlist=NULL;
}

int dfloatarray::add(floatlist* fl)
{
	floatlist_el* new_el=new floatlist_el;
	if(!new_el)
	{
		error(1);
		exit(1);
	}
	if(!&fl)
	{
		error(2);
		return 0;
	}
	if(!toplist)
                toplist=new_el;
        new_el->flval=fl;
        if(bottomlist)
                bottomlist->right=new_el;
        new_el->right=NULL;
        bottomlist=new_el;
	return 1;
}

floatlist& dfloatarray::operator[](int y)
{
        floatlist_el* current=toplist;
        int count=0;
        while(current)
        {
                if(count==y)
                        return *current->flval;
                current=current->right;
                count++;
        }
        error(3,(float)y);
        floatlist* retval=new floatlist(0);
        return *retval;
}

int dfloatarray::entries() const
{
	floatlist_el* current=toplist;
        int count=0;
        while(current)
        {
                count++;
                current=current->right;
        }
        return count;
}

int dfloatarray::operator=(dfloatarray& farr)
{
	int rows=farr.entries();
	reset();
	for(int i=0; i<rows; i++)
		add(new floatlist(farr[i]));		
	return 1;
}

int dfloatarray::operator==(dfloatarray& farr)
{
        int rows=farr.entries();
        if(rows!=entries())
                return 0;
        for(int i=0; i<rows; i++)
        {
                if((*this)[i]!=farr[i])
                        return 0;
        }
        return 1;
}

int dfloatarray::operator!=(dfloatarray& farr)
{
	if(*this==farr)
		return 0;
	return 1;
}

void dfloatarray::report(ostream& strm) const
{
	floatlist_el* current=toplist; 
        while(current) 
        { 
		current->flval->report(strm);
		strm<<"\n";
        current=current->right; 
        } 
}

void dfloatarray::error(int n, float val) const
{
	cout<<"ERROR CLASS dfloatarray ::";
        switch(n) {
        case 1: cout<<"add(): lack of memory!"<<endl; break;
	case 2: cout<<"add(): list to be added was NULL!"<<endl;break;
        case 3: cout<<"[]: index "<< (int)val<< " out of range: ";
                cout<<"[0.."<<entries()<<"]!"<<endl;
                break;
        default: cout << "error(): no error et all?" << endl;
        }
}

//******************************************************************************

dfloatcube::dfloatcube()
{
        frontarray=NULL;
        backarray=NULL;
}

dfloatcube::dfloatcube(int x, int y, int z)
{
        frontarray=NULL;
        backarray=NULL;
        for(int i=0; i<z; i++)
                add(dfloatarray(x,y));
}

dfloatcube::~dfloatcube()
{
        reset();
}

void dfloatcube::reset()
{
        floatarray_el* current=frontarray;
        floatarray_el* next;
        while(current)
        {
                next=current->right;
                current->faval.reset();
                delete current;
                current=next;
        }
        frontarray=NULL;
        backarray=NULL;
}

int dfloatcube::add(dfloatarray fa)
{
        floatarray_el* new_el=new floatarray_el;
        if(!new_el)
        {
                error(1);
                exit(1);
        }
        if(!&fa)
        {
                error(2);
                return 0;
        }
        if(!frontarray)
                frontarray=new_el;
        new_el->faval=fa;
        if(backarray)
                backarray->right=new_el;
        new_el->right=NULL;
        backarray=new_el;
        return 1;
}
 
dfloatarray& dfloatcube::operator[](int z)
{
        floatarray_el* current=frontarray;
        int count=0;
        while(current)
        {
                if(count==z)
                        return current->faval;
                current=current->right;
                count++;
        }
        error(3,(float)z);
        dfloatarray* retval=new dfloatarray;
        return *retval;
}

int dfloatcube::entries() const
{
        floatarray_el* current=frontarray;
        int count=0;
        while(current)
        {
                count++;
                current=current->right;
        }
        return count;
}
 
int dfloatcube::operator=(dfloatcube& fcube)
{
        int layers=fcube.entries();
        reset();
        for(int i=0; i<layers; i++)
                add(dfloatarray(fcube[i]));
        return 1;
}
 
int dfloatcube::operator==(dfloatcube& fcube)
{
        int layers=fcube.entries();
        if(layers!=entries())
                return 0;
        for(int i=0; i<layers; i++)
        {
                if((*this)[i]!=fcube[i])
                        return 0;
        }
        return 1;
}
 
int dfloatcube::operator!=(dfloatcube& fcube)
{
        if(*this==fcube)
                return 0;
        return 1;
}
 
void dfloatcube::report(ostream& strm) const
{
        floatarray_el* current=frontarray;
        int layer=0;
        while(current)
        {
                strm<<"LAYER: "<<layer<<endl;
                current->faval.report(strm);
                strm<<endl;
                layer++;
                current=current->right;
        }
}
void dfloatcube::error(int n, float val) const
{
        cout<<"ERROR CLASS dfloatcube :: ";
        switch(n) {
        case 1: cout<<"add(): lack of memory!"<<endl; break;
        case 2: cout<<"add(): list to be added was NULL!"<<endl;break;
        case 3: cout<<"[]: index "<< (int)val<< " out of range: ";
                cout<<"[0.."<<entries()<<"]!"<<endl;
                break;
        default: cout << "error(): no error et all?" << endl;
        }
}
 
;

#endif
