/*FOCUS-2.0 is a sampling and statistical data analysis program
for spatial data sets. Copyright (C) 2003  Lutz Tischendorf

This program is free software; you can redistribute it
and/or modify it under the terms of the GNU General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public
License along with this program; if not, write to the Free
Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.*/

#ifndef __CONSTRAINT_CPP
#define __CONSTRAINT_CPP

#include "constraint.h"
#include "logging.h"
#include "focusregistry.h"
#include <algorithm>

Constraint::Constraint(const char* _name, const char* _class_name)
{
	name = _name;
	class_name = _class_name;
}

Constraint::~Constraint()
{
}

bool Constraint::operator==(const Constraint& _constraint) const
{
	return  name == _constraint.name &&
			class_name == _constraint.class_name;
}

bool Constraint::operator!=(const Constraint& _constraint) const
{
	return !((*this)==_constraint);
}
	
bool Constraint::operator <(const Constraint& _constraint) const
{
	Logging::log_error("Constraint::operator < should only be called on derived classes",NULL);
	return true;
}

void Constraint::report(ostream& strm) const
{
	strm<<class_name<<"::"<<name;
}

DistanceConstraint::DistanceConstraint(const char* _name) : Constraint(_name, "DistanceConstraint")
{
	distance = NULL_DOUBLE;
	sampling_type = INSIDE;
	multiplier = 1;
}

DistanceConstraint::DistanceConstraint(const char* _name, double _distance, double _multiplier, sample_types _sampling_type) 
					: Constraint(_name, DistanceConstraint::CLASS_NAME)
{
	distance = _distance;
	sampling_type = _sampling_type;
	multiplier = _multiplier;
}

DistanceConstraint::~DistanceConstraint()
{
}

bool DistanceConstraint::operator==(const DistanceConstraint& _constraint) const
{
	return	name == _constraint.name && 
			class_name == _constraint.class_name &&
			distance == _constraint.distance &&
			multiplier == _constraint.multiplier;
}

bool DistanceConstraint::operator <(const DistanceConstraint& _constraint) const
{
	return distance < _constraint.distance;
}

void DistanceConstraint::report(ostream& strm) const
{
	Constraint::report(strm);
	strm<<"(distance="<<distance<<", multiplier="<<multiplier<<")\n";
}

const char* DistanceConstraint::CLASS_NAME = "DistanceConstraint";

ConstraintList::ConstraintList()
{
	sort_status = UNSORTED;
}

ConstraintList::~ConstraintList()
{
	ConstraintList::iterator constraint_iterator = begin();
	while(constraint_iterator != end())
	{ 
		Constraint* tmpConstraint = *constraint_iterator++;
		delete tmpConstraint;
	}
	clear();
}
	
void ConstraintList::addConstraint(Constraint* _constraint)
{
	push_back(_constraint);
}

void ConstraintList::sortAscending()
{
	if(sort_status != ASCENDING)
	{
		sort(begin(), end(), Less_Constraint());
		sort_status = ASCENDING;
	}
}

void ConstraintList::sortDescending()
{
	if(sort_status != DESCENDING)
	{
		sort(begin(), end(), Greater_Constraint());
		sort_status = DESCENDING;
	}
}

Constraint* ConstraintList::getConstraintAtIndex(int _index)
{
	return (*this)[_index];
}

void ConstraintList::report(ostream& strm) const
{
	ConstraintList::const_iterator constraint_iterator = begin();
	while(constraint_iterator != end())
		(*constraint_iterator++)->report(strm);
}

#endif