/*FOCUS-2.0 is a sampling and statistical data analysis program
for spatial data sets. Copyright (C) 2003  Lutz Tischendorf

This program is free software; you can redistribute it
and/or modify it under the terms of the GNU General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public
License along with this program; if not, write to the Free
Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.*/

#ifndef __OUTPUTWRITER_CPP
#define __OUTPUTWRITER_CPP

#include "outputwriter.h"
#include "focusregistry.h"
#include "focusutil.h"
#include "logging.h"
#include "observer.h"
#include "constraint.h"
#include "controller.h"
#include "sampler.h"
#include "plot.h"
#include <io.h>
#include <math.h>
#include <vector>

#if !defined (STLPORT) || defined(_STLP_USE_NAMESPACES)
using namespace std;
#endif

OutputWriter::OutputWriter(FocusRegistry* _registry, char* _base_name, bool _time_stamp, bool _write_plots, bool _write_stats)
{
	registry = _registry;
	registry->registerOutputWriter(this);
	
	raw_stats_stream = NULL;
	aggr_stats_stream = NULL;
	plot_stream = NULL;

	base_name = _base_name;
	raw_stats_file_name = NULL;
	aggr_stats_file_name = NULL;
	plot_file_name = NULL;

	write_aggr_header = true;
	write_stats_header = true;
	write_statistics = _write_stats;
	write_plots = _write_plots;
	time_stamp = _time_stamp;

	init_outputfile_names();
	init_output_streams();
}

void OutputWriter::init_outputfile_names()
{
	Logging::log_debug("OutputWriter::init_outputfile_names()...",NULL);

	char* ds;
	if(time_stamp)
		ds = Utility::get_date_string();
	else
		ds = "";

	if(write_statistics)
	{
		raw_stats_file_name = new char[50];
		aggr_stats_file_name = new char[50];
		sprintf(raw_stats_file_name,"%s%s%s%s",base_name,"_raw",ds,".csv");
		sprintf(aggr_stats_file_name,"%s%s%s%s",base_name,"_aggr",ds,".csv");
	}
	if(write_plots)
	{
		plot_file_name = new char[50];
		sprintf(plot_file_name,"%s%s%s%s",base_name,"_plots",ds,".csv");
	}

	Logging::log_debug("OutputWriter::init_outputfile_names() DONE.",NULL);
}

void OutputWriter::init_output_streams() throw (FocusException)
{
	Logging::log_debug("OutputWriter::init_output_streams()...",NULL);

	if(!write_statistics && !write_plots)
	{
		char input_char = ' ';

		cout <<"\n\nYou set 'Statistics' = no and 'WritePlots' = no. No ouput will be written!\n";
		cout <<"	Do you want want to proceed (y/n)? "; cin>>input_char;
    
		if(input_char == 'n') 
		{ 
			cout<<"\n"<<"Model terminated."<<"\n"; 
			throw FocusException("OutputWriter::init_output_streams() user exit!");
		} 
	}

	if(write_statistics)
	{
		if(access(raw_stats_file_name,02) == 0)
   			confirmFileName(raw_stats_file_name);
		if(access(aggr_stats_file_name,02) == 0)
   			confirmFileName(aggr_stats_file_name);

		FILE* dummyfile = fopen(raw_stats_file_name,"w");
		if(!dummyfile)
			throw FocusException("OutputWriter::init_output_streams() '", raw_stats_file_name, "' could not be opened!");
		fclose(dummyfile);
   
		dummyfile = fopen(aggr_stats_file_name,"w");
		if(!dummyfile)
			throw FocusException("OutputWriter::init_output_streams() '", aggr_stats_file_name, "' could not be opened!");
		fclose(dummyfile);

		raw_stats_stream = new ofstream(raw_stats_file_name, ios::out);
		if(!raw_stats_stream)
			throw FocusException("OutputWriter::init_output_streams() '", raw_stats_file_name, "' could not be opened!");
		raw_stats_stream->setf(ios::scientific, ios::fixed);
		raw_stats_stream->precision(10);

		aggr_stats_stream = new ofstream(aggr_stats_file_name, ios::out);
		if(!aggr_stats_stream)
			throw FocusException("OutputWriter::init_output_streams() '", aggr_stats_file_name, "' could not be opened!");
		aggr_stats_stream->setf(ios::scientific, ios::fixed);
		aggr_stats_stream->precision(10);
	}
	if(write_plots)
	{
		if(access(plot_file_name,02) == 0)
   			confirmFileName(plot_file_name);	

		FILE* dummyfile = fopen(plot_file_name,"w");
		if(!dummyfile)
			throw FocusException("OutputWriter::init_output_streams() '", plot_file_name, "' could not be opened!");
		fclose(dummyfile);
		
		plot_stream = new ofstream(plot_file_name, ios::out);
		if(!plot_stream)
			throw FocusException("OutputWriter::init_output_streams() '", plot_file_name, "' could not be opened!");
		plot_stream->setf(ios::scientific, ios::fixed);
		plot_stream->precision(10);
	}

	Logging::log_debug("OutputWriter::init_output_streams() DONE.",NULL);
}

void OutputWriter::confirmFileName(char* _filename) throw (FocusException)
{
	char input_char = ' ';

	cout <<"\n\n'"<<_filename<<"' already exists!"<<"\n";
	cout<<"	Do you want to overwrite it (y/n)? "; cin>>input_char;
    
	if(input_char != 'y' && input_char != 'Y') 
	{ 
		cout<<"\n"<<"Model terminated."<<"\n"; 
		throw FocusException("OutputWriter::confirmFileName(",_filename, ") user exit!");
	} 
	else 
		return;
}

OutputWriter::~OutputWriter()
{
	if(raw_stats_file_name)
		delete raw_stats_file_name;
	if(aggr_stats_file_name)
		delete aggr_stats_file_name;
	if(plot_file_name)
		delete plot_file_name;

	close_output_streams();
}

void OutputWriter::close_output_streams()
{
	Logging::log_debug("OutputWriter::close_output_streams()...",NULL);

	if(raw_stats_stream)
	{
		raw_stats_stream->close();
		delete raw_stats_stream;
		raw_stats_stream = NULL;
	}
	if(aggr_stats_stream)
	{
		aggr_stats_stream->close();
		delete aggr_stats_stream;
		aggr_stats_stream = NULL;
	}
	if(plot_stream)
	{
		plot_stream->close();
		delete plot_stream;
		plot_stream = NULL;
	}

	Logging::log_debug("OutputWriter::close_output_streams() DONE.",NULL);
}

void OutputWriter::write_outputfile_names(ostream& strm)
{
	if(write_statistics || write_plots)
		cout<<"The results have been written into the following files:\n\n";

	if(write_statistics)
	{
		strm<<raw_stats_file_name<<"\n";
		strm<<aggr_stats_file_name<<"\n";
	}
	if(write_plots)
	{
		strm<<plot_file_name<<"\n";
	}

	cout<<"\n\n";
}

void OutputWriter::write_iteration_header(ostream& strm)
{
	strm<<"RUN,";
}

void OutputWriter::write_constraint_header(ostream& strm)
{
	strm<<"CONSTR,S_TYPE,C_VALUE,C_MULTIP,C_MVALUE,";
}

void OutputWriter::write_regression_header(ostream& strm)
{
	strm<<"M,B,";
}

void OutputWriter::write_anova_header(ostream& strm)
{
	strm<<"#PLOTS,SST,SSR,SSE,MSE,F,R-SQUARE,R,";
}

void OutputWriter::write_duplicate_header(ostream& strm)
{
	strm<<"DUPLIC,";
}

void OutputWriter::write_unique_header(ostream& strm)
{
	strm<<"UNIQUE,";
}

void OutputWriter::write_plot_header(ostream& strm)
{
	Sampler* sampler = registry->getSampler();
	if(sampler)
	{
		int plots = sampler->getMaxPlots();
		for (int i=1; i<=plots; i++)
			strm<<"P_"<<i<<",";
	}
	else
		Logging::log_debug("OutputWriter::write_plot_header() no sampler available!",NULL);
}

void OutputWriter::write_plot_data()
{
	if(write_plots)
	{
		Logging::log_debug("OutputWriter::write_plot_data()...",NULL);

		cout<<"\n\nWriting plot data ";

		ObserverList* observerlist = registry->getObserverList();
		ObserverList::iterator obs_iterator = observerlist->begin();
		while(obs_iterator != observerlist->end())
		{
			if(!strcmp((*obs_iterator)->getName(),"PlotObserver"))
			{
				(*obs_iterator)->report_variable_names(*plot_stream);
				(*obs_iterator)->report(*plot_stream);
				break;
			}
		}

		cout<<"\n";

		Logging::log_debug("OutputWriter::write_plot_data() DONE.",NULL);
	}
}

void OutputWriter::write_raw_statistics(int _run)
{
	if(write_statistics)
	{
		Logging::log_debug("OutputWriter::write_raw_statistics(",Utility::int_to_char(_run),")...",NULL);

		if(write_stats_header)
		{
			write_iteration_header(*raw_stats_stream);
			write_duplicate_header(*raw_stats_stream);
			write_unique_header(*raw_stats_stream);
			write_constraint_header(*raw_stats_stream);
			write_regression_header(*raw_stats_stream);
			write_anova_header(*raw_stats_stream);
			write_plot_header(*raw_stats_stream);
			(*raw_stats_stream)<<"\n";
			write_stats_header = false;
		}
		write_iteration(_run, *raw_stats_stream);
		write_duplicate(*raw_stats_stream);
		write_unique(*raw_stats_stream);
		write_constraint(*raw_stats_stream);
		write_regression(*raw_stats_stream);
		write_anova(*raw_stats_stream);
		write_plot_ids(*raw_stats_stream);
		(*raw_stats_stream)<<"\n";
		
		Logging::log_debug("OutputWriter::write_raw_statistics(",Utility::int_to_char(_run),") DONE.",NULL);
	}	
}

void OutputWriter::write_aggregated_statistics()
{
	if(write_statistics)
	{
		Logging::log_debug("OutputWriter::write_aggregated_statistics()...",NULL);

		ObserverList* observerlist = registry->getObserverList();
		ObserverList::iterator obs_iterator = observerlist->begin();

		if(write_aggr_header)
		{
			write_constraint_header(*aggr_stats_stream);
			while(obs_iterator != observerlist->end())
			{
				if(strcmp((*obs_iterator)->getName(),"PlotObserver"))
				{
					(*obs_iterator)->report_variable_names(*aggr_stats_stream);
				}
				obs_iterator++;
			}
			(*aggr_stats_stream)<<"\n";
			write_aggr_header = false;
		}
		write_constraint(*aggr_stats_stream);

		obs_iterator = observerlist->begin();
		while(obs_iterator != observerlist->end())
		{
			if(strcmp((*obs_iterator)->getName(),"PlotObserver"))
			{
				(*obs_iterator)->report(*aggr_stats_stream);
			}
			obs_iterator++;
		}
		(*aggr_stats_stream)<<"\n";

		Logging::log_debug("OutputWriter::write_aggregated_statistics() DONE.",NULL);
	}
}

void OutputWriter::write_iteration(int _run, ostream& strm)
{
	strm<<_run<<",";
}

void OutputWriter::write_constraint(ostream& strm)
{
	Constraint* constraint = registry->getController()->getConstraint();
	if(constraint && constraint->getClass_name() == DistanceConstraint::CLASS_NAME)
	{
		DistanceConstraint* dist_constraint = (DistanceConstraint*)constraint;
		strm<<dist_constraint->getName()<<",";
		strm<<(dist_constraint->getSampleType() == INSIDE ? "INSIDE," : "OUTSIDE,");
		strm<<dist_constraint->getDistance()<<",";
		strm<<dist_constraint->getMultiplier()<<",";
		strm<<(dist_constraint->getDistance() * dist_constraint->getMultiplier())<<",";
	}
	else
	{
		strm<<"INVALID,INVALID,INVALID,INVALID,";
		Logging::log_debug("OutputWriter::write_constraint() no constraint data available!",NULL);
	}
}

void OutputWriter::write_regression(ostream& strm)
{
	RegressionData* reg_data = registry->getController()->getRegressionData();
	if(reg_data)
	{
		strm<<(reg_data->get_m(0) == NULL_DOUBLE ? "INVALID" : Utility::float_to_char(reg_data->get_m(0)))<<",";
		strm<<(reg_data->get_b() == NULL_DOUBLE ? "INVALID" : Utility::float_to_char(reg_data->get_b()))<<",";
	}
	else
	{
		strm<<"INVALID,INVALID,";
		Logging::log_debug("OutputWriter::write_regression() no regression data available!",NULL);
	}
}

void OutputWriter::write_anova(ostream& strm)
{
	AnovaData* anov_data = registry->getController()->getAnovaData();
	RegressionData* reg_data = registry->getController()->getRegressionData();

	if(anov_data && reg_data)
	{
		strm<<anov_data->get_n()<<",";
		strm<<anov_data->get_sst()<<",";
		strm<<anov_data->get_ssr()<<",";
		strm<<anov_data->get_sse()<<",";
		strm<<anov_data->get_mse()<<",";
		strm<<anov_data->get_f()<<",";
		strm<<anov_data->get_rr()<<",";
		strm<<anov_data->get_r()<<",";
	}
	else
	{

		strm<<registry->getController()->getNumberOfCurrentSelectedPlots()<<",INVALID,INVALID,INVALID,INVALID,INVALID,INVALID,INVALID,";
		Logging::log_debug("OutputWriter::write_anova() no anova data available!",NULL);
	}
}

void OutputWriter::write_duplicate(ostream& strm)
{
	Sampler* sampler = registry->getSampler();
	if(sampler)
	{
		strm<<(sampler->getAllowDuplicateSamples() == true ? "YES," : "NO,");
	}
	else
	{
		strm<<"INVALID,";
		Logging::log_debug("OutputWriter::write_duplicate() no sampler available!",NULL);
	}
}

void OutputWriter::write_unique(ostream& strm)
{
	Sampler* sampler = registry->getSampler();
	if(sampler)
	{
		strm<<(sampler->getEnforceUniqueSamples() == true ? "YES," : "NO,");
	}
	else
	{
		strm<<"INVALID,";
		Logging::log_debug("OutputWriter::write_unique() no sampler available!",NULL);
	}
}


void OutputWriter::write_plot_ids(ostream& strm)
{
	plotset_types plotset_type = registry->getController()->getPlotSetType();

	if(plotset_type == PLOT_BUFFER)
		registry->getPlotSetList()->getPlotSetForReferenceName(PS_PLOT_BUFFER)->getPlotList()->reportSelectedPlotIds(strm);
	else
	{
		PlotList* pred_plots = registry->getPlotSetList()->getPlotSetForReferenceName(PS_PLOT_PLOT_PREDICTOR)->getPlotList();
		PlotList* resp_plots = registry->getPlotSetList()->getPlotSetForReferenceName(PS_PLOT_PLOT_RESPONSE)->getPlotList();

		int selected_plots = registry->getController()->getNumberOfCurrentSelectedPlots();

		PlotList::iterator predplot_iterator = pred_plots->begin();
		PlotList::iterator respplot_iterator = resp_plots->begin();
		
		for(int i=0; i<selected_plots; i++)
		{
			while(!(*predplot_iterator)->isSelected() && predplot_iterator != pred_plots->end())
				predplot_iterator++;

			while(!(*respplot_iterator)->isSelected() && respplot_iterator != resp_plots->end())
				respplot_iterator++;

			if((*predplot_iterator)->isSelected() && (*respplot_iterator)->isSelected())
			{
				strm<<"P_"<<(*predplot_iterator)->getId();
				strm<<"/R_"<<(*respplot_iterator)->getId();
				strm<<",";
			}

			predplot_iterator++;
			respplot_iterator++;
		}
	}
}

#endif