/*FOCUS-2.0 is a sampling and statistical data analysis program
for spatial data sets. Copyright (C) 2003  Lutz Tischendorf

This program is free software; you can redistribute it
and/or modify it under the terms of the GNU General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public
License along with this program; if not, write to the Free
Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.*/

#ifndef __DISTANCE_CPP
#define __DISTANCE_CPP

#include "distance.h"
#include "defs.h"
#include "focusregistry.h"
#include <math.h>
#include <algorithm>

Distance::Distance()
{
	distance = NULL_DOUBLE;
}

Distance::Distance(double _distance, Plot* _plot_1, Plot* _plot_2)
{
	distance = _distance;
	plot_1 = _plot_1;
	plot_2 = _plot_2;
}

Distance::Distance(Plot* _plot_1, Plot* _plot_2)
{
	distance = NULL_DOUBLE;
	plot_1 = _plot_1;
	plot_2 = _plot_2;
}

Distance::~Distance()
{
}

bool Distance::operator==(const Distance& _distance) const
{
	return distance == _distance.distance;
}

bool Distance::operator!=(const Distance& _distance) const
{
	return !((*this)==_distance);
}

bool Distance::operator<(const Distance& _distance) const
{
	return distance <= _distance.distance;
}

void Distance::calculateDistance() throw (FocusException)
{
	if(!plot_1 || !plot_2)
		throw FocusException("Distance::calculateDistance() Plot_1 or Plot_2 or both are not initialized.");
	
	if(plot_1->getX() == NULL_DOUBLE || plot_1->getY() == NULL_DOUBLE ||
	   plot_2->getX() == NULL_DOUBLE || plot_2->getY() == NULL_DOUBLE)
	   throw FocusException("Distance::calculateDistance() Coordinates of Plot_1 or Plot_2 or both are not initialized.");

	if(distance != NULL_DOUBLE)
		Logging::log_error("Distance::calculateDistance() distance was already set. Will overwrite!",NULL);

	double x_dist = plot_1->getX() - plot_2->getX();
	double y_dist = plot_1->getY() - plot_2->getY();
	distance = sqrt(pow(x_dist,2) + pow(y_dist,2));
}

void Distance::report(ostream& strm) const
{
	strm<<"distance: "<<distance;
	strm<<" plot_1: "<< (plot_1 != NULL ? plot_1->getId() : "NULL");
	strm<<" plot_2: "<< (plot_2 != NULL ? plot_2->getId() : "NULL");
	strm<<"\n";
}

DistanceList::DistanceList()
{
	sort_status = UNSORTED;
}

DistanceList::~DistanceList()
{	
	clear();
}

void DistanceList::addDistance(Distance& _distance)
{
	push_back(_distance);
	sort_status = UNSORTED;
}

void DistanceList::sortAscending()
{
	if(sort_status != ASCENDING)
	{
		sort(begin(), end(), Less_Distance());
		sort_status = ASCENDING;
	}
}

void DistanceList::sortDescending()
{
	if(sort_status != DESCENDING)
	{
		sort(begin(), end(), Greater_Distance());
		sort_status = DESCENDING;
	}
}

void DistanceList::calculateDistances() throw (FocusException)
{
	DistanceList::iterator distance_iterator = begin();
	while(distance_iterator != end())
		(*distance_iterator++).calculateDistance();
}

void DistanceList::unselectPlotsWithinDistance(Plot* _plot, double _distance)
{
	sortAscending();

	DistanceList::iterator distance_iterator = begin();
	while(distance_iterator != end() && (*distance_iterator).getDistance() <= _distance)
	{
		Plot* tmpPlot_1 = (*distance_iterator).getPlot_1();
		Plot* tmpPlot_2 = (*distance_iterator).getPlot_2();
		
		if(*_plot == *tmpPlot_1)
			tmpPlot_2->unSelect();
		else if(*_plot == *tmpPlot_2)
			tmpPlot_1->unSelect();

		distance_iterator++;
	}
}

void DistanceList::verifyPlotsWithinDistance(double _distance)
{
	DistanceList::iterator distance_iterator = begin();
	while(distance_iterator != end())
	{
		if((*distance_iterator).getDistance() <= _distance &&
		    (*distance_iterator).getPlot_1()->isSelected() && 
			(*distance_iterator).getPlot_2()->isSelected())
			cout<<"Error at Distance: "<< (*distance_iterator).getDistance();
		
		distance_iterator++;
	}
}

void DistanceList::unselectPlotsOutsideDistance(Plot* _plot, double _distance)
{
	sortDescending();

	DistanceList::iterator distance_iterator = begin();
	while(distance_iterator != end() && (*distance_iterator).getDistance() >= _distance)
	{
		Plot* tmpPlot_1 = (*distance_iterator).getPlot_1();
		Plot* tmpPlot_2 = (*distance_iterator).getPlot_2();
		
		if(*_plot == *tmpPlot_1)
			tmpPlot_2->unSelect();
		else if(*_plot == *tmpPlot_2)
			tmpPlot_1->unSelect();

		distance_iterator++;
	}
}

void DistanceList::verifyPlotsOutsideDistance(double _distance)
{
	DistanceList::iterator distance_iterator = begin();
	while(distance_iterator != end())
	{
		if((*distance_iterator).getDistance() >= _distance &&
		    (*distance_iterator).getPlot_1()->isSelected() && 
			(*distance_iterator).getPlot_2()->isSelected())
			cout<<"Error at Distance: "<< (*distance_iterator).getDistance();
		
		distance_iterator++;
	}
}

void DistanceList::report(ostream& strm) const
{
	DistanceList::const_iterator distance_iterator = begin();
	while(distance_iterator != end())
		(*distance_iterator++).report(strm);
}

#endif