/*FOCUS-2.3 is a sampling and statistical data analysis program
for spatial data sets. Copyright (C) 2003  Lutz Tischendorf

This program is free software; you can redistribute it
and/or modify it under the terms of the GNU General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public
License along with this program; if not, write to the Free
Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.*/

#ifndef __CONTROLLER_CPP
#define __CONTROLLER_CPP

#include "observer.h"
#include "controller.h"
#include "inputreader.h"
#include "outputwriter.h"
#include "statistics.h"
#include "sampler.h"
#include "focusregistry.h"
#include "focusutil.h"
#include "plot.h"
#include <time.h>

Controller::Controller(FocusRegistry* _registry, const char* _config_filename)
{
	registry = _registry;
	registry->registerController(this);
	config_filename = _config_filename;
	iterations = 0;
	current_constraint = NULL;
	current_constraint_index = NULL_SHORT;
	current_regression = NULL;
	current_anova = NULL;
	current_predictor_values = NULL;	
	current_response_values = NULL;
	current_predictor_index = 0;
	statistics = true;
	fs_order = NULL_SHORT;
	fs_percentage = NULL_SHORT;
	functional_stability = false;
	zero_regression = false;
	include_zero_regression = true;
	current_max_order = NULL_SHORT;
	plotset_type = NO_TYPE;
}

void Controller::execute()
{
	init();
	
	const char* ps_type = (plotset_type == PLOT_BUFFER ? PS_PLOT_BUFFER : PS_PLOT_PLOT_RESPONSE);
	ConstraintList* constraints = registry->getPlotSetList()->getPlotSetForReferenceName(ps_type)->getConstraints();
	int number_of_constraints = constraints->size();

	cout<<"\n\n";

	for(current_constraint_index=1; current_constraint_index<=number_of_constraints; current_constraint_index++)
	{
		current_constraint = (DistanceConstraint*)constraints->getConstraintAtIndex(current_constraint_index-1);
		registry->getSampler()->resetSampleHistory();
		
		cout<<"Processing constraint: "<<Utility::float_to_char(((DistanceConstraint*)current_constraint)->getDistance())<<" ";
		int dot_gap = iterations/(iterations<10?iterations:10);
		for(int i=1; i<=iterations; i++)
		{
			if(i==dot_gap)
			{
				dot_gap+=iterations/(iterations<10?iterations:10);
				cout<<".";
			}

			sampleByDistance((DistanceConstraint*)current_constraint);

			if(statistics)
			{
				regression();
				anova();	

				if(functional_stability && Utility::randint() < fs_percentage)
					functional_stability_test();
				
				write_raw_stats(i);
			}	
			observe(i-1);	
			resetCurrentData();
		}
		calc_aggr_stats();
		write_aggr_stats();
		reset_observers();
		cout<<"\n";
	}

	write_plot_data();
	closeOutputStreams();

	cout<<"\n\nFocus2.3.1 terminated successfully!\n\n";
	writeOutputFileNames();
}

void Controller::init()
{
	cout<<"FOCUS version 2.3.1 Copyright (C) 2007 LT\n";
	cout<<"FOCUS-2.3.1 comes with ABSOLUTELY NO WARRANTY!\n";

	cout<<"\nInitializing FOCUS 2.3.1 ";
	
	srand( (unsigned)time( NULL ) );

	registry->registerInputReader(new InputReader(registry, config_filename != NULL ? config_filename : "FocusConfig.xml"));
	registry->registerStatistics(new Statistics(registry));

	cout<<".";

	InputReader* reader = registry->getInputReader();

	iterations = reader->getNumberOfIterations();
	include_zero_regression = reader->getIncludeZeroRegression();
	unsigned short maxPlots = reader->getMaximumPlotsPerIteration();
	bool duplicates = reader->getDuplicatesAllowedFlag();
	bool uniques = reader->getUniqueEnforceFlag();
	unsigned int duplicateRetrials = reader->getDuplicateRetrials();

	plotset_type = reader->getPlotSetType();
	ConstraintList* constraints = reader->getConstraints();
	
	const char* predictorVariableFileName = reader->getPredictorVariablesFileName();
	const char* responseVariableFileName = reader->getResponseVariablesFileName();
	
	if(plotset_type == PLOT_BUFFER)
	{
		const char* distanceFileName = reader->getDistanceMatrixFileName();
		reader->validatePlotBufferDataSet(distanceFileName,predictorVariableFileName,responseVariableFileName);
		
		reader->readPlotDistanceMatrix(distanceFileName);
		cout<<".";
		reader->readVariables(predictorVariableFileName, PREDICTOR, WITH_CONTEXT);
		cout<<".";
		reader->readVariables(responseVariableFileName, RESPONSE);
		
		registry->getPlotSetList()->getPlotSetForReferenceName(PS_PLOT_BUFFER)->setConstraints(constraints);
	}
	else if(plotset_type == PLOT_PLOT)
	{
		reader->readPlotVariables(predictorVariableFileName, PREDICTOR);
		cout<<".";
		reader->readPlotVariables(responseVariableFileName, RESPONSE);
		cout<<".";
		registry->getPlotSetList()->getPlotSetForReferenceName(PS_PLOT_PLOT_PREDICTOR)->setConstraints(constraints);
		registry->getPlotSetList()->getPlotSetForReferenceName(PS_PLOT_PLOT_RESPONSE)->setConstraints(constraints);
	}	
	
	registry->registerSampler(new Sampler(registry, maxPlots, duplicateRetrials, duplicates, uniques));

	statistics = reader->getStatistics();
	write_plots = reader->getWritePlots();
	if(write_plots)
		registry->addObserver(new PlotObserver(registry, maxPlots, iterations, constraints->size()));

	char* base_name = (char*)reader->getOutputBaseName();
	bool time_stamp = reader->getTimeStamp();
	
	cout<<".";
	
	registry->registerOutputWriter(new OutputWriter(registry, base_name, time_stamp, write_plots, statistics));
	
	if(statistics)
			registry->addObserver(new AnovaObserver(registry, iterations));

	functional_stability = reader->getFunctionalStability();
	if(functional_stability)
	{
		fs_order = reader->getMaximumOrder();
		fs_percentage = reader->getPercentage();
		registry->addObserver(new FunctionalStabilityObserver(registry, fs_order, fs_percentage));
	}		
}

void Controller::observe(int _run)
{
	if(!zero_regression)
	{
		ObserverList* observerlist = registry->getObserverList();
		ObserverList::iterator obs_iterator = observerlist->begin();
		while(obs_iterator != observerlist->end())
			(*obs_iterator++)->observe(_run);
	}
}

void Controller::sampleByDistance(DistanceConstraint* _constraint)
{
	if(plotset_type == PLOT_BUFFER)
	{
		registry->getSampler()->sampleByDistance(_constraint->getAppliedDistance(), _constraint->getSampleType(),
												 PREDICTOR, PLOT_BUFFER);
	}
	else if(plotset_type == PLOT_PLOT)
	{
		registry->getSampler()->sampleByDistance(_constraint->getAppliedDistance(), _constraint->getSampleType(),
												 PREDICTOR, PLOT_PLOT);
		registry->getSampler()->sampleByDistance(_constraint->getAppliedDistance(), _constraint->getSampleType(),
												 RESPONSE, PLOT_PLOT);
	}
	set_current_variable_values();
}

void Controller::set_current_variable_values()
{
	double current_distance = ((DistanceConstraint*)current_constraint)->getDistance();
	sample_types current_sample_type = ((DistanceConstraint*)current_constraint)->getSampleType();
	
	if(plotset_type == PLOT_BUFFER)
	{
		PlotList* plot_list = registry->getPlotSetList()->getPlotSetForReferenceName(PS_PLOT_BUFFER)->getPlotList();

		current_predictor_index = plot_list->getPredictorVariableIndexByContext(current_distance, current_sample_type);

		current_predictor_values = plot_list->getPredictorVariableValuesForSelectedPlots(current_predictor_index);
		current_response_values = plot_list->getResponseVariableValuesForSelectedPlots(0);
		number_of_current_selected_plots = plot_list->getNumberOfSelectedPlots();
	}
	else if(plotset_type == PLOT_PLOT)
	{
		PlotList* pred_list = registry->getPlotSetList()->getPlotSetForReferenceName(PS_PLOT_PLOT_PREDICTOR)->getPlotList();
		PlotList* resp_list = registry->getPlotSetList()->getPlotSetForReferenceName(PS_PLOT_PLOT_RESPONSE)->getPlotList();
		
		current_predictor_values = pred_list->getPredictorVariableValuesForSelectedPlots(0);
		current_response_values = resp_list->getResponseVariableValuesForSelectedPlots(0);

		int sel_pred = pred_list->getNumberOfSelectedPlots();
		int sel_resp = resp_list->getNumberOfSelectedPlots();

		number_of_current_selected_plots = sel_pred < sel_resp ? sel_pred : sel_resp;
	}
}

void Controller::regression()
{	
	if(number_of_current_selected_plots < 2)
	{
		if(!include_zero_regression)
			zero_regression = true;
		else
			zero_regression = false;

		Logging::log_debug("Controller::regression() less than 2 plots selected. Regression skipped.",NULL);
		return;
	}

	current_regression = registry->getStatistics()->regression(current_predictor_values,
															   current_response_values,
															   number_of_current_selected_plots,1);
	
	if(!include_zero_regression && current_regression->get_b() == 0 && current_regression->get_m(0) == 0)
		zero_regression = true;
	else
		zero_regression = false;
	
}

void Controller::functional_stability_test()
{

	if(number_of_current_selected_plots < 2)
	{
		Logging::log_debug("Controller::functional_stability_test() less than 2 plots selected. FS skipped.",NULL);
		return;
	}
	if(!zero_regression)
	{
		current_max_order = registry->getStatistics()->functional_stability_test(current_predictor_values, 
																				 current_response_values, 
																				 number_of_current_selected_plots,
																				 fs_order);
	}
}

void Controller::anova()
{
	if(number_of_current_selected_plots < 2)
	{
		Logging::log_debug("Controller::anova() less than 2 plots selected. Anova skipped.",NULL);
		return;
	}
	if(!current_regression)
	{
		Logging::log_debug("Controller::anova() no regression data available. Anova skipped.",NULL);
		return;
	}
	if(!zero_regression)
	{
		double* yobs = current_regression->get_yobs();
		double* ypred = current_regression->get_ypred();
		int n = current_regression->get_n();
		float m_0 = current_regression->get_m(0);

		current_anova = registry->getStatistics()->anova(yobs, ypred, n, m_0);
	}
}

void Controller::write_raw_stats(int _run)
{
	if(!zero_regression)
		registry->getOutputWriter()->write_raw_statistics(_run);
}

void Controller::write_aggr_stats()
{
	registry->getOutputWriter()->write_aggregated_statistics();
}

void Controller::write_plot_data()
{
	registry->getOutputWriter()->write_plot_data();
}

void Controller::calc_aggr_stats()
{
	ObserverList* observerlist = registry->getObserverList();
	ObserverList::iterator obs_iterator = observerlist->begin();
	while(obs_iterator != observerlist->end())
		(*obs_iterator++)->calc_stats();
}

void Controller::resetCurrentData()
{
	if(current_regression)
		delete current_regression;
	current_regression = NULL;
	if(current_anova)
		delete current_anova;
	current_anova = NULL;
	if(current_predictor_values)
		delete current_predictor_values;
	current_predictor_values = NULL;
	if(current_response_values)
		delete current_response_values;
	current_response_values = NULL;
	current_max_order = NULL_SHORT;
}

void Controller::reset_observers()
{
	ObserverList* observerlist = registry->getObserverList();
	ObserverList::iterator obs_iterator = observerlist->begin();
	while(obs_iterator != observerlist->end())
	{
		(*obs_iterator)->reset();
		(*obs_iterator++)->init();
	}
}

void Controller::closeOutputStreams()
{
	registry->getOutputWriter()->close_output_streams();
}

void Controller::writeOutputFileNames()
{
	registry->getOutputWriter()->write_outputfile_names();
}

#endif