/*FOCUS-2.0 is a sampling and statistical data analysis program
for spatial data sets. Copyright (C) 2003  Lutz Tischendorf

This program is free software; you can redistribute it
and/or modify it under the terms of the GNU General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public
License along with this program; if not, write to the Free
Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.*/

#ifndef __CONSTRAINT_H
#define __CONSTRAINT_H

#include "defs.h"
#include "logging.h"
#include <iostream>
#include <vector>
#include <functional>

#if !defined (STLPORT) || defined(_STLP_USE_NAMESPACES)
using namespace std;
#endif

class Constraint
{
protected:
	const char*		name;			//name of the constraint, e.g. buffer width
	const char*		class_name;		//name of the class, e.g. BufferConstraint

public:
	Constraint(const char* _name, const char* _class_name);
	~Constraint();

	bool operator==(const Constraint& _constraint) const;
	bool operator!=(const Constraint& _constraint) const;
	virtual bool operator <(const Constraint& _constraint) const;

	void setName(const char* _name)				{ name = _name; };
	const char* getName() const					{ return name; };

	const char* getClass_name() const			{ return class_name; };

	void report(ostream& strm=cout) const;
	
	static const char* CLASS_NAME;
};

class DistanceConstraint : public Constraint
{
private:
	double			multiplier;
	double			distance;
	sample_types	sampling_type;

public:
	DistanceConstraint(const char* _name);
	DistanceConstraint(const char* _name, double _distance, double _multiplier, sample_types _sampling_type);
	~DistanceConstraint();

	bool operator==(const DistanceConstraint& _constraint) const;
	bool operator <(const DistanceConstraint& _constraint) const;

	void setDistance(double _distance)				{ distance = _distance; };
	double getDistance() const						{ return distance; };
	double getAppliedDistance() const				{ return distance * multiplier; };

	void setMultiplier(double _multiplier)			{ multiplier = _multiplier; };
	double getMultiplier() const					{ return multiplier; };

	void setSampleType(sample_types _sampling_type)	{ sampling_type = _sampling_type; };
	sample_types getSampleType() const				{ return sampling_type; }

	void report(ostream& strm=cout) const;	
};

class FocusRegistry;

class ConstraintList : public vector<Constraint*>
{
private:
	sort_types		sort_status;

public:
	ConstraintList();
	~ConstraintList();
	
	void addConstraint(Constraint* _constraint);
	void sortAscending();
	void sortDescending();
	Constraint* getConstraintAtIndex(int _index);

	sort_types getSort_status() const				{ return sort_status; };

	void report(ostream& strm=cout) const;
};

class Less_Constraint : public binary_function<Constraint*, Constraint*, bool> 
{
public:
	bool operator()(Constraint* _x, Constraint* _y) 
	{ 
		if(_x && !strcmp(_x->getClass_name(),"DistanceConstraint") &&
		   _y && !strcmp(_y->getClass_name(),"DistanceConstraint"))
			return (*((DistanceConstraint*)_x)) < (*((DistanceConstraint*)_y)); 
		else
		{
			Logging::log_error("Less_Constraint() type mismatch. Cannot compare ",
				               "different object types of Constraint!",NULL);
			return true;
		}
	}
};

class Greater_Constraint : public binary_function<Constraint*, Constraint*, bool> 
{
public:
	bool operator()(Constraint* _x, Constraint* _y) 
	{ 
		return !((*_x) < (*_y)); 
	}
};


#endif